#ifndef FELIXSERVER_BUFFEREDPUBLISHER_HPP
#define FELIXSERVER_BUFFEREDPUBLISHER_HPP

#include <cstddef>
#include <cstdint>
#include <string>
#include <span>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/NetioPublisher.hpp>
#include <felixbus/FelixBusWriter.hpp>

#include "felix-server/Definitions.hpp"

namespace felix_server {
  /**
   * @brief Class for publishing of data using buffered sending
   *
   * The publisher can be configured using the settings struct. The settings include the IP and port
   * ofnthe publisher, the network type (LIBFABRIC for RDMA or ASYNCMSG for TCP), the buffer size,
   * and number of buffers per sender, and a buffer timeout after which the buffer is flushed even
   * if it is not full. A timeout of 0 means that the buffer is flushed only when it is full.
   *
   * The parameters are written to the bus given the path, group, and filename.
   *
   * Two callbacks can be provided: One for subscription and one for unsubscription. These callbacks
   * are called when a subscription or unsubscription is received. The callbacks are called with the
   * tag and endpoint as parameters.
   *
   * @note The publisher is supposed to be created from the @ref FelixServer class and not directly.
   */
  class BufferedPublisher
  {
  public:
    struct Settings {
      std::string ip{};
      std::uint16_t port{};
      netio3::NetworkType network_type{};
      std::size_t buffer_size{};
      std::uint32_t buffer_count{};
      std::size_t buffer_timeout{};
      std::string bus_path{};
      std::string bus_group{};
      std::string bus_filename{};
      OnSubscriptionCallback on_sub{nullptr};
      OnUnsubscriptionCallback on_unsub{nullptr};
    };

    /**
     * @brief Constructor for the BufferedPublisher class
     *
     * Only data for the given tags can be published (only these tags are declared in the bus and
     * clients can therefore only subscribe to these tags).
     *
     * @param settings The settings for the publisher
     * @param evloop The event loop to use for the publisher
     * @param tags The tags to declare for the publisher
     */
    BufferedPublisher(Settings settings,
                      std::shared_ptr<netio3::BaseEventLoop> evloop,
                      std::span<const std::uint64_t> tags);

    /**
     * @brief Publish data using a span of iovec
     *
     * The data is published to all subscribers of the given tag. An optional status byte can be set
     * (default is OK).
     *
     * The data is provided as a span of iovec (so does not need to be contiguous).
     *
     * This function returns
     * * OK if the publish was successful to all subscribers.
     * * FAILED if at least one send operation failed (indicating a retry will not work)
     * * NO_SUBSCRIPTIONS if there are no subscribers for the given tag
     * * NO_RESOURCES if there are not enough resources to send the data. In this case the function
     *   must be called again with the same parameters except that retry is set to true. This must
     *   be done until it no longer returns NO_RESOURCES.
     *
     * @param tag The tag to publish to
     * @param data The data to publish
     * @param retry Whether this publish is a retry of a previous publish
     * @param user_status The user status to be put into the header
     * @return The status of the publish
     */
    [[nodiscard]] netio3::NetioPublisherStatus publish(std::uint64_t tag,
                                                       std::span<const iovec> data,
                                                       bool retry,
                                                       std::uint8_t user_status = 0);

    /**
     * @brief Publish contiguous data
     *
     * The data is provided as a span of bytes (has to be contiguous).
     *
     * See @ref publish for more details and how to handle the return values.
     *
     * @param tag The tag to publish to
     * @param data The data to publish
     * @param retry Whether this publish is a retry of a previous publish
     * @param user_status The user status to be put into the header
     * @return The status of the publish
     */
    [[nodiscard]] netio3::NetioPublisherStatus publish(std::uint64_t tag,
                                                       std::span<const std::uint8_t> data,
                                                       bool retry,
                                                       std::uint8_t user_status = 0);

  private:
    /**
     * @brief Declare the tags in the bus
     *
     * @param tags The tags to declare
     */
    void declare(std::span<const std::uint64_t> tags);

    Settings m_settings{};
    felixbus::FelixBusWriter m_bus_writer{};
    netio3::NetioPublisher m_publisher;
  };
}  // namespace felix_server

#endif  // FELIXSERVER_BUFFEREDPUBLISHER_HPP