#ifndef FELIXCLIENT_BLOCKDECODER_HPP
#define FELIXCLIENT_BLOCKDECODER_HPP

#include <cstdint>
#include <cstring>
#include <functional>
#include <span>

#include <ers/ers.h>

#include <felix/felix_client_thread.hpp>

#include "felix/Block.hpp"
#include "felix/BlockDecoderStats.hpp"
#include "felix/ChunkDecoder.hpp"

namespace felix {
  /**
   * @brief Block decoder
   *
   * This class decodes blocks received from the firmware. It handles the
   * fragmentation of messages into subchunks and calls the callback with the
   * decoded data. One block decoder is created for each FID.
   */
  class BlockDecoder
  {
  public:
    /**
     * @brief Constructor
     *
     * @param fid FID of the decoder
     * @param has_streams True if the block has stream identifiers
     * @param on_data_callback Callback to call when a message is decoded
     */
    explicit BlockDecoder(std::uint64_t fid,
                          bool has_streams,
                          std::function<void(std::uint64_t fid,
                                             std::span<const std::uint8_t> data,
                                             std::uint8_t status)> on_data_callback);

    /**
     * @brief Decode chunks from a block
     *
     * Invoke the on_data_callback for every decoded subchunk.
     *
     * @param block Block to decode
     */
    void decode(const Block& block);

    /**
     * @brief Get the statistics of the decoder
     *
     * @return Statistics of the decoder
     */
    [[nodiscard]] FelixClientThread::BlockDecoderStats get_statistics() const;

  private:
    /**
     * @brief Subchunk header
     *
     * The subchunk header is a 32 bit value with the following layout:
     * - 16 bits: length of the subchunk
     * - 9 bits: unused
     * - 1 bit: busy mask
     * - 1 bit: CRC error
     * - 1 bit: error
     * - 1 bit: truncation
     * - 3 bits: type
     */
    union SubchunkHeader {
      struct {
        uint32_t length : 16;
        uint32_t unused : 9;
        uint32_t busymask : 1;
        uint32_t crcerr : 1;
        uint32_t err : 1;
        uint32_t trunc : 1;
        uint32_t type : 3;
      } __attribute__((packed)) data;
      uint32_t value;
    };

    /**
     * @brief Check if the block is valid
     *
     * Currently, only checks size and not sequence number.
     *
     * @param block Block to check
     * @return True if the block is valid, false otherwise
     */
    [[nodiscard]] bool check_block_integrity(const Block& block);

    std::uint64_t m_fid{};
    ChunkDecoder m_chunk_decoder;
    int m_last_seqnr{-1};
    BlockDecoderStatistics m_stats{};
  };
}  // namespace felix

#endif  // FELIXCLIENT_BLOCKDECODER_HPP