#ifndef FELIXCLIENT_BLOCKDECODERBUFFER_HPP
#define FELIXCLIENT_BLOCKDECODERBUFFER_HPP

#include <cstddef>
#include <cstdint>
#include <span>
#include <vector>

namespace felix {
  /**
   * @brief Buffer for the block decoder
   *
   * This class is used to store the data of a block while it is being decoded. It is used to handle
   * the fragmentation of messages into subchunks.
   */
  class BlockDecoderBuffer
  {
  public:
    /**
     * @brief Constructor
     *
     * @param fid FID of the decoder
     */
    explicit BlockDecoderBuffer(std::uint64_t fid) : m_fid{fid}
    {
      m_buffer.reserve(preallocated_max_msg_size);
    }

    /**
     * @brief Append data to the buffer
     *
     * @param data Data to append
     */
    void append(std::span<const std::uint8_t> data)
    {
      m_buffer.insert(m_buffer.end(), data.begin(), data.end());
    }

    /**
     * @brief Update the status byte
     *
     * ORs the status byte with the current status byte.
     *
     * @param status_byte Status bit to set
     */
    void update_status_byte(std::uint8_t status_byte) { m_status_byte |= status_byte; }

    /**
     * @brief Update the fid (used for stream identifiers)
     *
     * ORs the fid with the current fid.
     *
     * @param stream_id Stream ID to set
     */
    void update_fid(std::uint8_t stream_id) { m_fid |= stream_id; }

    /**
     * @brief Get the status byte
     *
     * @return Status byte
     */
    [[nodiscard]] std::uint8_t get_status_byte() const { return m_status_byte; }

    /**
     * @brief Get the buffer
     *
     * @return Buffer
     */
    [[nodiscard]] std::span<const std::uint8_t> get_buffer() const { return m_buffer; }

    /**
     * @brief Get the fid
     *
     * @return FID
     */
    [[nodiscard]] std::uint64_t get_fid() const { return m_fid; }

    /**
     * @brief Reset the buffer
     *
     * Clears the buffer and resets the status byte and fid.
     */
    void reset()
    {
      m_buffer.clear();
      m_status_byte = 0;
      constexpr static std::uint64_t fid_mask = 0xFFFFFFFFFFFFFF00;
      m_fid &= fid_mask;
    }

    /**
     * @brief Check if the buffer is empty
     *
     * @return True if the buffer is empty, false otherwise
     */
    [[nodiscard]] bool empty() const { return m_buffer.empty(); }

  private:
    constexpr static auto preallocated_max_msg_size = std::size_t{1024 * 1024};

    std::vector<std::uint8_t> m_buffer;
    std::uint64_t m_fid{};
    std::uint8_t m_status_byte{};
  };
}  // namespace felix

#endif  // FELIXCLIENT_BLOCKDECODERBUFFER_HPP