#ifndef FELIXCLIENT_BLOCKMESSAGEDECODER_HPP
#define FELIXCLIENT_BLOCKMESSAGEDECODER_HPP

#include <cstdint>
#include <cstring>
#include <functional>
#include <span>

#include <absl/container/flat_hash_map.h>

#include <felix/felix_client_thread.hpp>
#include <felixbus/FelixBusReader.hpp>

#include "felix/BlockDecoder.hpp"

namespace felix {
  /**
   * @brief Block message decoder
   *
   * This class handles the decoding of messages containing blocks. It uses the @ref BlockDecoder to
   * decode the blocks and calls the callback with the decoded data.
   *
   * One must first call @ref register_fid to register a FID for block decoding. After that, the
   * @ref decode_block function can be used to decode a received message.
   */
  class BlockMessageDecoder
  {
  public:
    /**
     * @brief Constructor
     *
     * @param on_data_callback Callback to call when a message is decoded
     */
    explicit BlockMessageDecoder(std::function<void(std::uint64_t fid,
                                                    std::span<const std::uint8_t> data,
                                                    std::uint8_t status)> on_data_callback);

    /**
     * @brief Register a FID for block decoding
     *
     * This function registers a FID for block decoding. If the FID already exists, it will be
     * overwritten.
     *
     * @param fid FID to register
     * @param has_streams True if the block has stream identifiers
     */
    void register_fid(std::uint64_t fid, bool has_streams);

    /**
     * @brief Unregister a FID from block decoding
     *
     * @param fid FID to unregister
     */
    void unregister_fid(std::uint64_t fid);

    /**
     * @brief Decode a message containing blocks
     *
     * First, decode the message into a block. Then, decode the block into subchunks and call the
     * callback with the decoded data.
     *
     * @param fid FID of the block
     * @param message Message to decode
     */
    void decode_block(std::uint64_t fid, std::span<const std::uint8_t> message);

    /**
     * @brief Get the statistics of the block message decoder
     *
     * This function returns the statistics of all registered block decoders.
     *
     * @return Statistics of all registered block decoders
     */
    [[nodiscard]] std::map<std::uint64_t, FelixClientThread::BlockDecoderStats> get_statistics() const;

  private:
    /**
     * @brief Decode the message into a @ref Block
     *
     * @param message Message to decode
     * @return Block or std::nullopt if the message is not valid
     */
    [[nodiscard]] std::optional<Block> decode_message(std::span<const std::uint8_t> message) const;

    std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>
      m_on_data_callback;
    absl::flat_hash_map<std::uint64_t, std::unique_ptr<BlockDecoder>> m_block_decoder;
    mutable std::mutex m_mutex;
  };
}  // namespace felix

#endif  // FELIXCLIENT_BLOCKMESSAGEDECODER_HPP