#ifndef FELIXCLIENT_RECONNECTTIMER_HPP
#define FELIXCLIENT_RECONNECTTIMER_HPP

#include <chrono>
#include <cstdint>
#include <map>
#include <set>

#include <netio3-backend/EventLoop/EventTimerHandle.hpp>

namespace internal {
  class SubscriptionManager;
  class SenderManager;

  /**
   * @brief Timer to reconnect subscriptions and send connections
   *
   * In case of a connection loss (for example because of a server restart), connections are added
   * to this timer to re-establish the connection as soon as possible. The @ref SenderManager and
   * @ref SubscriptionManager add items to this timer when they notice a lost connection. When the
   * first item is added, the timer is started. On every tick, it will try to re-open the
   * connection. If the connection is established, the item is removed from the timer. If the
   * attempt failed it will be tried again.
   */
  class ReconnectTimer
  {
  public:
    /**
     * @brief Constructor
     *
     * @param evloop Event loop
     * @param interval Interval between reconnection attempts
     */
    ReconnectTimer(const std::shared_ptr<netio3::BaseEventLoop>& evloop,
                   std::chrono::milliseconds interval);

    /**
     * @brief Add subscriptions to the timer
     *
     * @param subscriber Subscription manager handling the subscription
     * @param tags Tags to reconnect
     */
    void add_subscription(SubscriptionManager* subscriber, const std::set<std::uint64_t>& tags);

    /**
     * @brief Add send connections to the timer
     *
     * @tparam R Type of range containing the tags
     * @param sender Sender manager handling the send connection
     * @param tags Tags to reconnect
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void add_send_connections(SenderManager* sender, const R& tags)
    {
      {
        std::lock_guard lock{m_mutex};
        if (not m_send_connections.contains(sender)) {
          m_send_connections.try_emplace(sender);
        }
        for (const auto tag : tags) {
          m_send_connections.at(sender).emplace(tag);
          m_state.try_emplace(tag, State::open);
        }
        check_and_start();
      }
    }

    /**
     * @brief Mark an attempt to reconnect a tag as failed
     *
     * Causes the timer to try to reconnect the tag again.
     *
     * @param tag Tag
     */
    void reset_state(std::uint64_t tag);

    /**
     * @brief Stop the reconnection attempts for all tags of one manager
     *
     * Called when a subscription is destructed to stop the reconnection attempts.
     *
     * @param subscriber Subscription manager
     */
    void remove_subscription(SubscriptionManager* subscriber);

    /**
     * @brief Stop the reconnection attempts for all tags of one sender
     *
     * Called when a sender is destructed to stop the reconnection attempts.
     *
     * @param sender Sender manager
     */
    void remove_send_connection(SenderManager* sender);

    /**
     * @brief Stop the reconnection attempts for a set of tags of one subscriber
     *
     * Called when resubscription was successful.
     *
     * @param subscriber Subscription manager
     * @param tags Tags to stop the reconnection attempts for
     */
    void remove_subscription(SubscriptionManager* subscriber, std::uint64_t tag);

    /**
     * @brief Stop the reconnection attempts for a set of tags of one sender
     *
     * Called when the connection was established.
     *
     * @tparam R Type of range containing the tags
     * @param sender Sender manager
     * @param tags Tags to stop the reconnection attempts for
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void remove_send_connections(SenderManager* sender, const R& tags)
    {
      std::lock_guard lock{m_mutex};
      for (const auto tag : tags) {
        m_state.erase(tag);
        m_send_connections.at(sender).erase(tag);
      }
      check_and_stop();
    }

    /**
     * @brief Stop the timer
     */
    void stop();

  private:
    /**
     * @brief Initiate reconnection attempts
     *
     * Subscribes/opens send connections for all pending items. Checks afterwards if no items are
     * pending anymore and stops if that is the case.
     */
    void on_timer();

    /**
     * @brief Start the timer if not running already
     */
    void check_and_start();

    /**
     * @brief Stop the timer if no more items are pending
     */
    void check_and_stop();

    enum class State {
      open,
      in_progress,
    };

    std::map<SubscriptionManager*, std::set<std::uint64_t>> m_subscriptions;
    std::map<SenderManager*, std::set<std::uint64_t>> m_send_connections;
    std::map<std::uint64_t, State> m_state;
    netio3::EventTimerHandle m_timer;
    std::chrono::milliseconds m_interval;
    mutable std::mutex m_mutex;
  };
}  // namespace internal

#endif  // FELIXCLIENT_RECONNECTTIMER_HPP