#ifndef FELIXCLIENT_SENDERWRAPPER_HPP
#define FELIXCLIENT_SENDERWRAPPER_HPP

#include <memory>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/NetioSender.hpp>

#include "felix/Netio3BusInterface.hpp"

namespace internal {
  enum class SenderMode {
    buffered,
    unbuffered,
  };
  enum class ConnectionStatus {
    connected,
    connecting,
    disconnecting,
    disconnected,
    refused,
  };

  /**
   * @brief Connection parameters for the @ref SenderWrapper
   */
  struct SenderParams {
    netio3::ConnectionParameters conn_params;
    ConnectionType connection_type{};

    constexpr auto operator<=>(const SenderParams&) const = default;
  };

  /**
   * @brief Wrapper around the netio3::NetioSender class for a given set of connection parameters.
   *
   * Keeps track of the connection status for each endpoint address and adds connection for
   * automatic reconnection if the connection is lost.
   */
  class SenderWrapper
  {
  public:
    /**
     * @brief Callbacks to the @ref SenderManager
     */
    struct CallbackConfig {
      std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
        on_connection_established;
      std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
        on_connection_closed;
      std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
        on_connection_refused;
      std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
        on_connection_lost_cb;
    };

    /**
     * @brief Constructor
     *
     * @param sender_params Connection parameters
     * @param evloop Event loop
     * @param thread_safety_model The thread safety model to use for the subscribers
     * @param callbacks Callbacks to the @ref SenderManager
     */
    SenderWrapper(const SenderParams& sender_params,
                      const std::shared_ptr<netio3::BaseEventLoop>& evloop,
                      netio3::ThreadSafetyModel thread_safety_model,
                      CallbackConfig callbacks);

    /**
     * @brief Open a connection to the given endpoint address
     *
     * @throws std::logic_error if the connection already exists
     * @param address Endpoint address
     */
    void open_connection(const netio3::EndPointAddress& address);

    /**
     * @brief Close a connection to the given endpoint address
     *
     * @throws std::logic_error if the connection does not exist
     * @param address Endpoint address
     */
    void close_connection(const netio3::EndPointAddress& address);

    /**
     * @brief Get the number of connections
     *
     * @return Number of connections
     */
    [[nodiscard]] std::size_t get_num_connections() const;

    /**
     * @brief Get the connection status for the given endpoint address
     *
     * @param address Endpoint address
     * @return Connection status
     */
    [[nodiscard]] ConnectionStatus get_connection_status(
      const netio3::EndPointAddress& address) const;

    /**
     * @brief Send data to the given endpoint address
     *
     * @throws felix::SendBeforeConnected if the connection is not established
     * @throws std::logic_error if the sender mode is unknown
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param address Endpoint address
     * @param fid FID
     * @param data Data to send
     * @param mode Sender mode
     * @param flush Flush the buffer
     */
    void send_data(const netio3::EndPointAddress& address,
                   std::uint64_t fid,
                   std::span<const uint8_t> data,
                   SenderMode mode,
                   bool flush = false);

    /**
     * @brief Send data to the given endpoint address
     *
     * Sends messages in a single transaction even if unbuffered mode is used if possible.
     *
     * @throws felix::SendBeforeConnected if the connection is not established
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param address Endpoint address
     * @param fid FID
     * @param data Data to send
     * @param mode Sender mode
     * @param flush Flush the buffer
     */
    void send_data(const netio3::EndPointAddress& address,
                   std::uint64_t fid,
                   std::span<const std::span<const uint8_t>> data,
                   SenderMode mode,
                   bool flush = false);

    /**
     * @brief Get the number of available buffers per FID
     *
     * Returns the minimum number of buffers since the last call to this function per FID.
     *
     * @return Number of buffers per FID
     */
    [[nodiscard]] std::vector<netio3::BufferStats> get_num_buffers();

  private:
    /**
     * @brief Handle the connection established event
     *
     * Update connection status and call callback to @ref SenderManager
     *
     * @param address Endpoint address
     */
    void on_connection_established(const netio3::EndPointAddress& address);

    /**
     * @brief Handle the connection refused event
     *
     * Update connection status and call callback to @ref SenderManager
     *
     * @param address Endpoint address
     */
    void on_connection_refused(const netio3::EndPointAddress& address);

    /**
     * @brief Handle the connection closed event
     *
     * Update connection status and call callback to @ref SenderManager. If the connection was not
     * supposed to be closed call the connection lost callback.
     *
     * @param address Endpoint address
     */
    void on_connection_closed(const netio3::EndPointAddress& address);

    /**
     * @brief Throw the appropriate exception if a send failed
     *
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param address Endpoint address
     */
    static void handle_send_status(netio3::NetioStatus status);

    SenderParams m_sender_params;
    netio3::NetioSender m_sender;
    std::map<netio3::EndPointAddress, ConnectionStatus> m_connection_status;
    std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
      m_on_connection_established_cb;
    std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
      m_on_connection_closed_cb;
    std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
      m_on_connection_refused_cb;
    std::function<void(const SenderParams&, const netio3::EndPointAddress&)>
      m_on_connection_lost_cb;
    mutable std::mutex m_mutex;

    constexpr static auto FLUSH_INTERVAL = std::uint64_t{5000};
  };
}  // namespace internal

#endif  // FELIXCLIENT_SENDERWRAPPER_HPP