#ifndef FELIXCLIENT_SUBSCRIBERWRAPPER_HPP
#define FELIXCLIENT_SUBSCRIBERWRAPPER_HPP

#include <chrono>
#include <condition_variable>
#include <functional>
#include <memory>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventSignalHandle.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/BufferFormatter.hpp>
#include <netio3/NetioSubscriber.hpp>
#include <netio3/Types.hpp>

#include "felix/Netio3BusInterface.hpp"

namespace internal {
  /**
   * @brief Set of parameters defining a single connection
   *
   * All subscriptions sharing this set of parameters will be handled by the same connection.
   */
  struct SubscriptionParams {
    netio3::EndPointAddress address;
    ConnectionType connection_type{};
    std::uint64_t num_pages{};
    std::uint64_t page_size{};
    bool has_streams{};
    bool sends_blocks{};
  };

  /**
   * @brief Set of callbacks to be called when certain events occur
   */
  struct SubscriptionCallbackConfig {
    std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>
      on_data_cb{nullptr};
    std::function<void(std::span<const std::uint8_t> data)> on_buffer_cb{nullptr};
    std::function<void(std::uint64_t)> on_subscription_cb{nullptr};
    std::function<void(std::uint64_t)> on_unsubscription_cb{nullptr};
    std::function<void(const std::set<std::uint64_t>&)> on_connection_lost_cb{nullptr};
    std::function<void(std::uint64_t)> on_resubscription_cb{nullptr};
  };

  /**
   * @brief Wrapper around the NetioSubscriber class to handle subscriptions for one connection type
   *
   * Every connection type requires its own instance of a NetioSubscriber. This class wraps around
   * the NetioSubscriber and provides functionality to subscribe and unsubscribe ind a synchronous
   * and asynchronous way. It also offers an API in terms of FIDs instead of EndPointAddresses.
   *
   * Provides callbacks on data, subscription, unsubscription, connection loss and resubscription.
   *
   * Subscriptions can have different states: Subscribing, subscribed, unsubscribing, resubscribing,
   * and timed out. When a subscription is requested the state will be set to subscribing. Once it
   * succeeds, it goes to subscribed. If the subscription failed it will be deleted. If the request
   * was synchronous and the subscription did not succeed within the timeout but also did not fail
   * it will be marked as timed out. If it succeeds in the future it will immediately unsubscribe
   * again. When unsubscribe is called the state goes to unsubscribing and once it succeeds it will
   * be deleted. If a subscription is lost (for example because the server went down) the state goes
   * to resubscribing and it will periodically attempt to resubscribe using the @ref ReconnectTimer
   * managed by the @ref SubscriptionManager.
   *
   * Failures to subscribe and unsubscribe will be communicated through exceptions. If it is marked
   * as failed, another attempt is not expected to succeed without further work (e.g. fixing the
   * server). If it is marked as timed out, another attempt might succeed. For obvious reasons,
   * timed out can only be reported for synchronous subscriptions.
   *
   * If a synchronous call dioes not throw all links are subscribed (or unsubscribed) when it
   * returns. For asynchronous calls, the success for each individual link is reported by the
   * on_subscription_cb, and on_unsubscription_cb callbacks.
   *
   * Subscriptions are allowed for links that are not yet subscribed, in subscribingor timed out
   * state. Subscriptions that are already established will report success, if they are in any other
   * state than those they will be marked as failed.
   *
   * Unsubscriptions are allowed for links in any state. If a subscribing link is unsubscribed and
   * afterwards the subscription succeeds, it will be unsubscribed again.
   *
   * Error handling for failed unsubscriptions is difficult as a failed unsubscription indicates
   * that communication with the server is impossible which effectively means that you should be
   * unsubscribed as the connection was lost.
   */
  class SubscriberWrapper
  {
  public:
    /**
     * @brief Construct a new Subscriber Wrapper object
     *
     * @param type The connection type to use (RDMA or TCP)
     * @param evloop The event loop to use for the NetioSubscriber and signals
     * @param local_ip The local IP address to listen for connections
     * @param thread_safety_model The thread safety model to use for the subscribers
     * @param callbacks The set of callbacks to be called when certain events occur
     */
    SubscriberWrapper(ConnectionType type,
                      std::shared_ptr<netio3::BaseEventLoop> evloop,
                      const std::string& local_ip,
                      netio3::ThreadSafetyModel thread_safety_model,
                      SubscriptionCallbackConfig callbacks,
                      std::function<void(std::uint64_t)> manger_unsub_cb);

    /**
     * @brief Subscribe to a set of FIDs asynchronously
     *
     * If the subscription fails (for example if the request is ill formed) or no connection could
     * be established an exception is thrown. In case no resources are available it will
     * automatically try again.
     *
     * This function will not wait for completion of the subscription. Use the on_subscription_cb to
     * receive confirmation of the subscription. If the request could be sent but no acknowledgment
     * was received the request might be pending indefinitely.
     *
     * @throws felix::AsyncSubscriptionFailedError If the subscription fails
     * @param params The set of FIDs and their corresponding SubscriptionParams
     */
    void subscribe_nb(const std::map<std::uint64_t, SubscriptionParams>& params);

    /**
     * @brief Unsuscribe from a set of FIDs asynchronously
     *
     * If the unsubscription fails (for example because there is no subscription or sending the
     * message failed) an exception is thrown. In case no resources are available it will
     * automatically try again.
     *
     * This function will not wait for completion of the unsubscription. Use the
     * on_unsubscription_cb to receive confirmation of the unsubscription.
     *
     * @throws felix::AsyncUnsubscriptionFailedError If the unsubscription fails
     * @param fids The set of FIDs to unsubscribe from
     */
    void unsubscribe_nb(std::span<const std::uint64_t> fids);

    /**
     * @brief Subscribe to a set of FIDs synchronously
     *
     * This function will wait for the subscription to complete. If the subscription fails or times
     * out for any FID an exception is thrown. The exception will contain information about which
     * FIDs are successfully subscribed, which failed and which timed out. All FIDs that are
     * subscribed stay subscribed. If a timed out subscription succeeds in the future it will
     * automatically unsubscribe again.
     *
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @param params The set of FIDs and their corresponding SubscriptionParams
     * @param timeout The timeout for the subscription
     */
    void subscribe(const std::map<std::uint64_t, SubscriptionParams>& params,
                   std::chrono::milliseconds timeout);

    /**
     * @brief Unsubscribe from a set of FIDs synchronously
     *
     * This function will wait for the unsubscription to complete. If the unsubscription fails or
     * times out for any FID an exception is thrown. The exception will contain information about
     * which FIDs are successfully unsubscribed, which failed and which timed out.
     *
     * @throws felix::UnsubscriptionFailedError If the unsubscription fails
     * @param fids The set of FIDs to unsubscribe from
     * @param timeout The timeout for the unsubscription
     */
    void unsubscribe(std::span<const std::uint64_t> fids, std::chrono::milliseconds timeout);

    /**
     * @brief Unsubscribe from all FIDs asynchronously
     *
     * Stops all retries for subscriptions and all resubscription attempts. Then, unsubscribes
     * asynchronously from all subscribed FIDs. Ignores errors.
     */
    void unsubscribe_all();

    /**
     * @brief Check if a certain FID is subscribed
     *
     * @param fid The FID to check
     * @return true If the FID is subscribed
     */
    [[nodiscard]] bool has_subscription(std::uint64_t fid) const;

    /**
     * @brief Get the number of subscribed FIDs
     *
     * @return std::size_t The number of subscribed FIDs
     */
    [[nodiscard]] std::size_t get_num_subscriptions() const;

  private:
    /**
     * @brief Filter subscription requests
     *
     * This function filters the subscription parameters into three sets: The set of FIDs that need
     * to be subscribed, the set of FIDs that are already subscribed and the set of FIDs that should
     * not be attempted to subscribe because they are currently unsubscribing.
     *
     * @param params The set of FIDs and their corresponding SubscriptionParams
     * @return std::tuple<std::map<std::uint64_t, SubscriptionParams>,
     *                    std::vector<std::uint64_t>,
     *                    std::vector<std::uint64_t>> The filtered sets
     */
    [[nodiscard]] std::tuple<std::map<std::uint64_t, SubscriptionParams>,
                             std::vector<std::uint64_t>,
                             std::vector<std::uint64_t>>
    filter_subscription_params(const std::map<std::uint64_t, SubscriptionParams>& params) const;

    /**
     * @brief Register a set of FIDs to be subscribing
     *
     * Updates internal trackers.
     *
     * @param params The set of FIDs and their corresponding SubscriptionParams
     */
    void register_subscriptions(const std::map<std::uint64_t, SubscriptionParams>& params);

    /**
     * @brief Subscribe to a set of FIDs
     *
     * Calls the subscribe function of the NetioSubscriber for each FID and returns the results.
     * Also registers successful FIDs that send blocks with block message decoder.
     *
     * @param params The set of FIDs and their corresponding SubscriptionParams
     * @return std::map<std::uint64_t, netio3::NetioStatus> The results of the subscription attempt
     */
    [[nodiscard]] std::map<std::uint64_t, netio3::NetioStatus> do_subscribe(
      const std::map<std::uint64_t, SubscriptionParams>& params);

    /**
     * @brief Unsubscribe from a set of FIDs
     *
     * Calls the unsubscribe function of the NetioSubscriber for each FID and returns the results.
     *
     * @param fids The set of FIDs to unsubscribe from
     * @return std::map<std::uint64_t, netio3::NetioStatus> The results of the unsubscription
     * attempt
     */
    [[nodiscard]] std::map<std::uint64_t, netio3::NetioStatus> do_unsubscribe(
      std::span<const std::uint64_t> fids);

    /**
     * @brief Get the FIDs whose subscription timed out
     *
     * This does not include FIDs that failed to subscribe.
     *
     * @param requested The set of FIDs that were requested to be subscribed
     * @param failed_fids The set of FIDs that failed to subscribe
     * @return std::vector<std::uint64_t> The set of FIDs that timed out
     */
    [[nodiscard]] std::vector<std::uint64_t> get_timed_out_subscriptions(
      const std::map<std::uint64_t, SubscriptionParams>& requested, const std::vector<std::uint64_t>& failed_fids) const;

    /**
     * @brief Get the FIDs whose unsubscribtion timed out
     *
     * This does not include FIDs that failed to unsubscribe.
     *
     * @param requested The set of FIDs that were requested to be unsubscribed
     * @param failed_fids The set of FIDs that failed to unsubscribe
     * @return std::vector<std::uint64_t> The set of FIDs that timed out
     */
    [[nodiscard]] std::vector<std::uint64_t> get_timed_out_unsubscriptions(
      std::span<const std::uint64_t> requested,
      const std::vector<std::uint64_t>& failed_fids) const;

    /**
     * @brief Get the FIDs that were successfully subscribed
     *
     * @param requested The set of FIDs that were requested to be subscribed
     * @return std::vector<std::uint64_t> The set of FIDs that were successfully subscribed
     */
    [[nodiscard]] std::vector<std::uint64_t> get_successful_subscriptions(
      const std::map<std::uint64_t, SubscriptionParams>& requested) const;

    /**
     * @brief Get the FIDs that were successfully unsubscribed
     *
     * @param requested The set of FIDs that were requested to be unsubscribed
     * @return std::vector<std::uint64_t> The set of FIDs that were successfully unsubscribed
     */
    [[nodiscard]] std::vector<std::uint64_t> get_successful_unsubscriptions(
      std::span<const std::uint64_t> requested) const;

    /**
     * @brief Collect FIDs that failed to unsubscribe and throw an exception
     *
     * @throws felix::AsyncUnsubscriptionFailedError with the list of failed FIDs
     * @param status_map The results of the unsubscription attempt
     */
    static void handle_unsubscription_failure(
      const std::map<std::uint64_t, netio3::NetioStatus>& status_map);

    /**
     * @brief Adds FIDs that returned NO_RESOURCES during unsubscription to the retry list
     *
     * @param status_map The results of the unsubscription attempt
     */
    void handle_unsubscription_timeout(const std::map<std::uint64_t, netio3::NetioStatus>& status_map);

    /**
     * @brief Retry subscriptions and unsubscriptions that failed due to NO_RESOURCES
     *
     * Executed by a timer if any requests returned NO_RESOURCES. Removes links from the list if
     * they succeeded (in a sense of not returning NO_RESOURCES). Stops when all requests succeeded.
     */
    void retry_again();

    /**
     * @brief Get the FIDs that are currently subscribed
     *
     * Only subscribed links, not subscribing or any other states.
     *
     * @return std::vector<std::uint64_t> The set of FIDs that are currently subscribed
     */
    [[nodiscard]] std::vector<std::uint64_t> get_subscribed_fids() const;

    /**
     * @brief Callback for data received
     *
     * Calls on_data callback that was provided in the constructor.
     *
     * @param msg The message that was received
     */
    void on_data_cb(const netio3::BufferMsg& msg);

    /**
     * @brief Callback for successful subscription
     *
     * Depending on state:
     * - SUBSCRIBING: Set state to SUBSCRIBED and notify waiting threads, call on_subscription_cb if
     * request was asynchronous
     * - RESUBSCRIBING: Mark as SUBSCRIBED and call on_resubscription_cb (to remove from reconnect
     * timer)
     * - TIMEDOUT: Call unsubscribe
     * - UNSUBSCRIBING: Call unsubscribe again
     * - SUBSCRIBED: Do nothing
     *
     * @param fid The FID that was successfully subscribed
     */
    void on_subscription_cb(std::uint64_t fid);

    /**
     * @brief Callback for successful unsubscription
     *
     * Depending on state:
     * - UNSUBSCRIBED: Do nothing
     * - UNSUBSCRIBING: Remove and notify waiting threads, call on_unsubscription_cb if request was
     * asynchronous
     * - Anything but UNSUBSCRIBING: Mark as RESUBSCRIBING, call on_connection_lost_cb (to add to
     * reconnect timer)
     *
     * @param fid The FID that was successfully unsubscribed
     */
    void on_unsubscription_cb(std::uint64_t fid);

    /**
     * @brief Callback for lost subscriptions
     *
     * Calls on_connection_lost_cb that was provided in the constructor (to add links to reconnect
     * timer). Set the state to RESUBSCRIBING.
     *
     * @param fids The set of FIDs that were lost
     */
    void on_subscription_lost_cb(const std::set<uint64_t>& fids);

    enum class SubscriptionState
    {
      SUBSCRIBING,
      SUBSCRIBED,
      UNSUBSCRIBING,
      RESUBSCRIBING,
      TIMEDOUT
    };

    std::shared_ptr<netio3::BaseEventLoop> m_evloop;
    netio3::NetioSubscriber m_subscriber;
    netio3::EventTimerHandle m_again_retry_timer;
    std::map<std::uint64_t, SubscriptionState> m_subscription_state;
    std::set<std::uint64_t> m_call_sub_cb_fids;
    std::set<std::uint64_t> m_call_unsub_cb_fids;
    std::map<std::uint64_t, netio3::EndPointAddress> m_subscribed_fids_to_address;
    std::map<std::uint64_t, SubscriptionParams> m_retry_subscribe_fids;
    std::vector<std::uint64_t> m_retry_unsubscribe_fids;
    constexpr static std::chrono::milliseconds m_retry_interval{100};
    std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>
      m_on_data_cb;
    std::function<void(std::span<const std::uint8_t> data)> m_on_buffer_cb;
    std::function<void(std::uint64_t)> m_on_subscription_cb;
    std::function<void(std::uint64_t)> m_on_unsubscription_cb;
    std::function<void(const std::set<std::uint64_t>&)> m_on_connection_lost_cb;
    std::function<void(std::uint64_t)> m_on_resubscription_cb;
    std::function<void(std::uint64_t)> m_manager_unsub_cb;
    std::condition_variable m_cv_sub;
    std::condition_variable m_cv_unsub;
    bool m_check_sub{false};
    bool m_check_unsub{false};
    mutable std::mutex m_mutex;
  };
}  // namespace internal

#endif  // FELIXCLIENT_SUBSCRIBERWRAPPER_HPP