#ifndef FELIXCLIENT_SUBSCRIPTIONMANAGER_HPP
#define FELIXCLIENT_SUBSCRIPTIONMANAGER_HPP

#include <chrono>
#include <functional>
#include <memory>
#include <optional>

#include <felix/felix_client_thread.hpp>
#include <felixbus/FelixBusReader.hpp>
#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventSignalHandle.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/BufferFormatter.hpp>
#include <netio3/NetioSubscriber.hpp>

#include "felix/BlockMessageDecoder.hpp"
#include "felix/Netio3BusInterface.hpp"
#include "felix/ReconnectTimer.hpp"
#include "felix/Settings.hpp"
#include "felix/SubscriberWrapper.hpp"

namespace internal {
  /**
   * @brief Manages multiple subscribers and their subscriptions
   *
   * Holds up to four @ref SubscriberWrapper instances, one for each connection type (RDMA and TCP)
   * and one for each decoding type (messages and blocks). Details about the subscription process
   * are described in the @ref SubscriberWrapper class.
   *
   * Subscriptions can be request for sets of FIDs. If all links are already subscribed an exception
   * is thrown, if at least a single link is not yet subscribed the request will be accepted and
   * handled by the @ref SubscriberWrapper instances.
   *
   * If you subscribe to links of different connection types subscribing might take up to 2 times
   * the timeout value, as the subscriptions are done sequentially for each connection type.
   *
   * The user can set callbacks when data is received, when subscriptions or unsubscriptions
   * succeeded (only required for asynchronous operations) and when subscriptions are lost and
   * re-established (optional).
   */
  class SubscriptionManager
  {
  public:
    /**
     * @brief Construct a new Subscription Manager object
     *
     * @param bus_settings The settings for the Felix bus (e.g. path)
     * @param evloop The event loop to use for the subscribers and signals/timers
     * @param local_ip The local IP address to use for the subscribers
     * @param thread_safety_model The thread safety model to use for the subscribers
     * @param reconnect_timer The reconnect timer if connections are lost
     * @param callbacks The callbacks (see above for description)
     */
    SubscriptionManager(const BusSettings& bus_settings,
                        std::shared_ptr<netio3::BaseEventLoop> evloop,
                        std::string local_ip,
                        netio3::ThreadSafetyModel thread_safety_model,
                        ReconnectTimer& reconnect_timer,
                        const SubscriptionCallbackConfig& callbacks);

    /**
     * @brief Destroy the Subscription Manager object
     *
     * Remove itself from @ref ReconnectTimer
     */
    ~SubscriptionManager();
    SubscriptionManager(const SubscriptionManager&) = delete;
    SubscriptionManager(SubscriptionManager&&) = delete;
    SubscriptionManager& operator=(const SubscriptionManager&) = delete;
    SubscriptionManager& operator=(SubscriptionManager&&) = delete;

    /**
     * @brief Subscribe to a set of FIDs asynchronously
     *
     * See @ref SubscriberWrapper::subscribe_nb for more details
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::AsyncSubscriptionFailedError If the subscription fails
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fids The FIDs to subscribe to
     */
    void subscribe_nb(std::span<const std::uint64_t> fids);

    /**
     * @brief Subscribe to a set of FIDs synchronously
     *
     * See @ref SubscriberWrapper::subscribe for more details
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fids The FIDs to subscribe to
     * @param timeout The timeout for the operation
     */
    void subscribe(std::span<const std::uint64_t> fids, std::chrono::milliseconds timeout);

    /**
     * @brief Unsubscribe from a set of FIDs asynchronously
     *
     * See @ref SubscriberWrapper::unsubscribe_nb for more details
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @throws felix::AsyncUnsubscriptionFailedError If the unsubscription fails
     * @param fids The FIDs to unsubscribe from
     */
    void unsubscribe_nb(std::span<const std::uint64_t> fids);

    /**
     * @brief Unsubscribe from a set of FIDs synchronously
     *
     * See @ref SubscriberWrapper::unsubscribe for more details
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @throws felix::UnsubscriptionFailedError If the subscription fails
     * @param fids The FIDs to unsubscribe from
     * @param timeout The timeout for the operation
     */
    void unsubscribe(std::span<const std::uint64_t> fids, std::chrono::milliseconds timeout);

    /**
     * @brief Unsubscribe from all FIDs asynchronously
     *
     * Simply delete subscribers. Connections will be dropped. Does not follow the unsubscription
     * protocol.
     */
    void unsubscribe_all_nb();

    /**
     * @brief Unsubscribe from all FIDs synchronously
     *
     * Actually send unsubscribe messages to the subscribers. Wait for the timeout to expire or
     * until all subscriptions are removed. Issue an error if subscriptions are still open.
     *
     * @param timeout The timeout for the operation
     */
    void unsubscribe_all(std::chrono::milliseconds timeout);

    /**
     * @brief Get the number of subscriptions
     *
     * @return std::size_t The number of subscriptions
     */
    [[nodiscard]] std::size_t get_num_subscriptions() const;

    /**
     * @brief Check if a specific FID is subscribed
     *
     * @param fid The FID to check
     * @return true If the FID is subscribed
     * @return false If the FID is not subscribed
     */
    [[nodiscard]] bool has_subscription(std::uint64_t fid) const;

    /**
     * @brief Get the statistics of the block message decoder
     *
     * This function returns the statistics of all registered block decoders.
     *
     * @return Statistics of all registered block decoders
     */
    [[nodiscard]] std::map<std::uint64_t, FelixClientThread::BlockDecoderStats> get_block_decoder_stats() const;

  private:
    struct SubscriberWrapperKey {
      ConnectionType connection_type{};
      bool blocks{};

      constexpr auto operator<=>(const SubscriberWrapperKey&) const = default;
    };

    /**
     * @brief Check which links need to be subscribed, group them by wrapper and do
     * bookkeeping
     *
     * Checks if a new @ref SubscriberWrapper is required and creates it if necessary.
     *
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fids The FIDs to subscribe to
     * @return std::map<SubscriberWrapperKey, std::vector<std::uint64_t>> Params per wrapper
     */
    [[nodiscard]] std::map<SubscriberWrapperKey, std::map<std::uint64_t, SubscriptionParams>>
    register_subscriptions(std::span<const std::uint64_t> fids);

    /**
     * @brief Subscribe to a set of FIDs
     *
     * Calls synchronous or asynchronous subscribe methods on the @ref SubscriberWrapper instances.
     *
     * @param requests The FIDs to subscribe to grouped by subscriber wrapper with parameters
     * @param timeout The timeout for the operation (async it std::nullopt)
     */
    void do_subscribe(
      const std::map<SubscriberWrapperKey, std::map<std::uint64_t, SubscriptionParams>>& requests,
      const std::optional<std::chrono::milliseconds>& timeout);

    /**
     * @brief Bookkeeping for unsubscriptions
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @param fids The FIDs to unsubscribe from
     * @return std::map<SubscriberWrapperKey, std::vector<std::uint64_t>> FIDs per wrapper
     */
    [[nodiscard]] std::map<SubscriberWrapperKey, std::vector<std::uint64_t>>
      register_unsubscriptions(std::span<const std::uint64_t> fids);

    /**
     * @brief Unsubscribe from a set of FIDs
     *
     * Calls synchronous or asynchronous unsubscribe methods on the @ref SubscriberWrapper
     * instances.
     *
     * @param requests The FIDs to unsubscribe from grouped by wrapper
     * @param timeout The timeout for the operation (async it std::nullopt)
     */
    void do_unsubscribe(const std::map<SubscriberWrapperKey, std::vector<std::uint64_t>>& requests,
                        const std::optional<std::chrono::milliseconds>& timeout);

    /**
     * @brief Get the information about a specific FID from the bus
     *
     * @throws felix::BusException If the information is not available in the bus
     * @param fid The FID to get the information for
     * @return SubscriptionParams The information about the FID
     */
    [[nodiscard]] SubscriptionParams get_info_bus(std::uint64_t fid) const;

    /**
     * @brief Get the information about a set of FIDs from the bus
     *
     * Calls @ref get_info_bus for each FID in the set.
     *
     * @throws felix::BusException If the information is not available in the bus
     * @param fids The FIDs to get the information for
     * @return std::map<std::uint64_t, SubscriptionParams> The information about the FIDs
     */
    [[nodiscard]] std::map<std::uint64_t, SubscriptionParams> get_info_per_fid(
      std::span<const std::uint64_t> fids) const;

    /**
     * @brief Get the wrapper key for a specific FID that is already managed
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @param fid The FID to get the information for
     * @return SubscriberWrapperKey The wrapper key of the FID
     */
    [[nodiscard]] SubscriberWrapperKey get_info_existing(std::uint64_t fid) const;

    /**
     * @brief Map a set of FIDs that are already managed to their wrappers
     *
     * Calls @ref get_info_existing for each FID in the set.
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @param fids The FIDs to get the information for
     * @return std::map<SubscriberWrapperKey, std::vector<std::uint64_t>> FIDs per wrapper
     */
    [[nodiscard]] std::map<SubscriberWrapperKey, std::vector<std::uint64_t>> get_info_groups_existing(
      std::span<const std::uint64_t> fids) const;

    /**
      * @brief Get the number of subscriptions
      *
      * @return std::size_t The number of subscriptions
      */
    [[nodiscard]] std::size_t do_get_num_subscriptions() const;

    /**
     * @brief Check if a specific FID is subscribed
     *
     * @param fid The FID to check
     * @return true If the FID is subscribed
     * @return false If the FID is not subscribed
     */
    [[nodiscard]] bool do_has_subscription(std::uint64_t fid) const;

    /**
     * @brief Callback for successful subscription
     *
     * Clean up resources in block decoder.
     *
     * @param fid The FID that was successfully subscribed
     */
    void on_unsubscription(std::uint64_t fid);

    /**
     * @brief Cleanup the subscription tracker
     *
     * Remove the FID from the tracker.
     *
     * @param fid The FID to remove
     */
    void cleanup_subscription_tracker(std::uint64_t fid);

    /**
     * @brief Callback for resubscription
     *
     * Remove the subscription from the reconnect timer and call the user callback.
     *
     * @param fid The FID that was resubscribed
     */
    void on_resubscription(std::uint64_t fid);

    /**
     * @brief Callback for connection loss
     *
     * Add the subscriptions to the reconnect timer and call the user callback.
     *
     * @param fids The FIDs that were lost
     */
    void on_connection_lost(const std::set<std::uint64_t>& fids);

    mutable felixbus::FelixBusReader m_bus;
    std::shared_ptr<netio3::BaseEventLoop> m_evloop;
    std::string m_local_ip;
    netio3::ThreadSafetyModel m_thread_safety_model{};
    std::reference_wrapper<ReconnectTimer> m_reconnect_timer;
    std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>
      m_on_data_cb;
    std::function<void(std::span<const std::uint8_t> data)> m_on_buffer_cb;
    std::function<void(std::uint64_t)> m_on_subscription_cb;
    std::function<void(std::uint64_t)> m_on_unsubscription_cb;
    std::function<void(const std::set<std::uint64_t>&)> m_on_connection_lost_cb;
    std::function<void(std::uint64_t)> m_on_resubscription_cb;
    std::map<SubscriberWrapperKey, SubscriberWrapper> m_subscribers;
    std::map<std::uint64_t, SubscriberWrapper*> m_subscriber_by_fid;
    felix::BlockMessageDecoder m_block_decoder;
    mutable std::mutex m_mutex;
    mutable std::mutex m_mutex_subscribers;
  };
}  // namespace internal

#endif  // FELIXCLIENT_SUBSCRIPTIONMANAGER_HPP