#ifndef FELIXCLIENT_TRICKLEFORMATTER_HPP
#define FELIXCLIENT_TRICKLEFORMATTER_HPP

#include <string>
#include <vector>

#include "TrickleConfigContainer.hpp"

namespace trickle {

/**
 * @brief Class for formatting trickle messages
 *
 * This class provides methods for formatting trickle messages into JSON strings.
 */
class TrickleFormatter {

public:
    /**
     * @brief Initializes a new trickle configuration
     * 
     * Prepares the formatter for creating a new trickle configuration by resetting
     * the internal container state.
     */
    void create_config();
    
    /**
     * @brief Adds a message to the current trickle configuration
     *
     * @param fid The fid identifier
     * @param data The trickle configuration for one FE device in binary. Yet to be formatted
     *
     * Stores the provided message data in the internal container for later processing.
     */
    void append_config(const uint64_t fid, const std::vector<uint8_t>& data);

    /**
     * @brief Adds a message to the current trickle configuration
     *
     * @param fid The fid identifier
     * @param data The trickle configuration for one FE device as a string. Yet to be formatted.
     *
     * Stores the provided message data in the internal container for later processing.
     */
    void append_config(const uint64_t fid, const std::string& data);

    /**
     * @brief Formats all collected trickle messages into a JSON string
     * 
     * @return String containing the JSON-formatted trickle configuration
     */
    [[nodiscard]] std::string format_config();

    /**
     * @brief Formats a command to start trickle
     *
     * @return String containing the JSON-formatted start trickle command
     */
     [[nodiscard]] std::string format_start_command() const;

    /**
     * @brief Formats a command to stop trickle
     * 
     * @return String containing the JSON-formatted stop trickle command
     */
     [[nodiscard]] std::string format_stop_command() const;

    /**
     * @brief Formats a command to select BCIDs for trickle
     * 
     * @param first_bcid The first BCID in the range
     * @param last_bcid The last BCID in the range
     * @return String containing the JSON-formatted BCID selection command
     */
     [[nodiscard]] std::string format_select_bcids_command(uint32_t first_bcid, uint32_t last_bcid) const;

    /**
     * @brief Formats a command to select the throttle factor for trickle
     * 
     * @param throttle_factror slow the trickling by N times [1 = no throttle]
     * @return String containing the JSON-formatted BCID selection command
     */
     [[nodiscard]] std::string format_throttle_command(std::uint32_t throttle_factor) const;

    /**
     * @brief Check if the trickle configuration has been initialized
     *
     * @return true if initialize, false otherwise
     */
    [[nodiscard]] bool is_config_initialized() const;

private:
    TrickleConfigContainer m_container; ///< Container for storing trickle messages
    bool m_initialized{}; ///< Flag to check if trickle config  has been initialized
};

} // namespace trickle

#endif // FELIXCLIENT_TRICKLEFORMATTER_HPP
