#ifndef FELIXBUSFS_CALLBACKTIMER_HPP
#define FELIXBUSFS_CALLBACKTIMER_HPP

#include <chrono>
#include <condition_variable>
#include <functional>
#include <mutex>
#include <thread>

namespace felixbus {
  /**
   * @brief A class for calling a function periodically
   *
   * Used to touch or update bus files periodically.
   */
  class CallBackTimer
  {
  public:
    CallBackTimer() = default;
    CallBackTimer(const CallBackTimer&) = delete;
    CallBackTimer& operator=(const CallBackTimer&) = delete;
    CallBackTimer(CallBackTimer&&) = delete;
    CallBackTimer& operator=(CallBackTimer&&) = delete;

    /**
     * @brief Destructor
     *
     * Stops the timer.
     */
    ~CallBackTimer() { stop(); }

    /**
     * @brief Stop the timer
     */
    void stop()
    {
      if (is_running()) {
        m_thread.request_stop();
        m_thread.join();
      }
    }

    /**
     * @brief Start the timer
     *
     * @param period The period to call the function
     * @param func The function to call
     */
    void start(std::chrono::milliseconds period, const std::function<void(void)>& func)
    {
      if (is_running()) {
        stop();
      }
      m_thread = std::jthread([period, func](const std::stop_token stop_token) {
        while (not stop_token.stop_requested()) {
          std::mutex mutex;
          std::unique_lock lock(mutex);  // cppcheck-suppress localMutex
          std::condition_variable_any().wait_for(lock, stop_token, period, [] { return false; });
          if (stop_token.stop_requested()) {
            break;
          }
          func();
        }
      });
    }

  private:
    /**
     * @brief Check if the timer is running
     *
     * @return true if the timer is running
     */
    [[nodiscard]] bool is_running() const noexcept
    {
      return m_thread.joinable() and not m_thread.get_stop_token().stop_requested();
    }

    std::jthread m_thread;
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_CALLBACKTIMER_HPP