#ifndef FELIXBUSFS_FELIXBUSFILE_HPP
#define FELIXBUSFS_FELIXBUSFILE_HPP

#include <filesystem>
#include <fstream>
#include <set>
#include <string>
#include <functional>

#include "felixbus/FelixBusInfo.hpp"
#include "felixbus/CallBackTimer.hpp"

using namespace std::chrono_literals;

/**
 * @brief A class managing FELIX bus information file operations
 *
 * This class handles writing and maintaining a single FELIX bus file. The get_current_info_func is
 * a function that can be provided to periodically update the bus file. If not set, the bus file
 * will still be touched periodically to mark it as not stale but it may not receive any updates. If
 * cleanup is set to true, the bus file and directories will be removed on destruction.
 */
namespace felixbus {
  class FelixBusFile
  {
  public:
    /**
     * @brief Construct a new FelixBusFile object
     *
     * @param path The path to the bus file
     * @param get_current_info_func A function that returns the current set of FelixBusInfo
     * @param timeout The timeout for acquiring a lock on the bus file
     * @param verbose Whether to print verbose output
     * @param cleanup Whether to cleanup the bus file and directories on destruction
     * @throws std::ios_base::failure if the file cannot be opened
     * @throws FailedGetMetadataException if the hostname or username cannot be retrieved
     */
    FelixBusFile(std::filesystem::path path,
                 std::function<std::set<FelixBusInfo>()> get_current_info_func,
                 std::chrono::milliseconds timeout,
                 bool verbose,
                 bool cleanup);

    /**
     * @brief Destroy the FelixBusFile object
     *
     * Close and remove the bus file if cleanup is set to true.
     */
    ~FelixBusFile();
    FelixBusFile(const FelixBusFile&) = delete;
    FelixBusFile& operator=(const FelixBusFile&) = delete;
    FelixBusFile(FelixBusFile&&) = delete;
    FelixBusFile& operator=(FelixBusFile&&) = delete;

    /**
     * @brief Write a FelixBusInfo to the bus file
     *
     * Appends the FelixBusInfo to the bus file. While writing, a write lock is acquired on the
     * file.
     *
     * @param info The FelixBusInfo to write
     * @throws std::ios_base::failure if the write operation fails
     * @throws std::system_error if the write lock cannot be acquired due to an OS error
     * @throws LockedException if the write lock cannot be acquired due to timeout
     */
    void write(const FelixBusInfo& info);

    /**
     * @brief Clear the bus file
     *
     * @throws std::ios_base::failure if the clear operation fails
     */
    void clear();

    constexpr static auto BUS_TOUCH_INTERVAL = 10s;

  private:
    /**
     * @brief Callback function for the timer
     *
     * If the get_current_info function is set, this function will check if the content has to be
     * updated and update it. Otherwise, touch the file.
     */
    void timer_callback();

    /**
     * @brief Touch the bus file
     *
     * Update the last write time of the bus file to the current time.
     */
    void touch();

    /**
     * @brief Update the bus file
     *
     * Clear the bus file and write the current set of FelixBusInfo to it.
     */
    void update();

    /**
     * @brief Release the bus file
     *
     * Close the bus file and remove it if cleanup is set to true. Also stops the timer. Swallows
     * all exceptions in case of failure.
     */
    void release();

    /**
     * @brief Get the hostname of the machine
     *
     * @return std::string The hostname
     * @throws FailedGetMetadataException if the hostname cannot be retrieved
     */
    [[nodiscard]] static std::string get_hostname();

    /**
     * @brief Get the username of the user
     *
     * @return std::string The username
     * @throws FailedGetMetadataException if the username cannot be retrieved
     */
    [[nodiscard]] static std::string get_username();

    std::function<std::set<FelixBusInfo>()> m_get_current_info;
    std::set<FelixBusInfo> m_info;
    std::filesystem::path m_path;
    std::ofstream m_file;
    std::chrono::milliseconds m_timeout;
    CallBackTimer m_timer;
    std::string m_hostname;
    pid_t m_pid{};
    std::string m_username;
    bool m_verbose{};
    bool m_cleanup{};
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_FELIXBUSFILE_HPP