#ifndef FELIXBUSFS_FELIXBUSREADER_HPP
#define FELIXBUSFS_FELIXBUSREADER_HPP

#include <filesystem>
#include <string>

#include <simdjson.h>

#include "felixbus/FelixBusInfo.hpp"

using namespace std::chrono_literals;

namespace felixbus {
  /**
   * @brief A class for reading FELIX bus information from files
   *
   * Reading requires locking the file to prevent writing while reading. If the file is locked, the
   * read operation will throw a LockedException.
   *
   * First, it tries to match the FID exactly. If that fails, it looks it masks the stream ID
   * and matches entries with stream=true.
   *
   * In both cases, if more than a match is found the last one is returned.
   */
  class FelixBusReader
  {
  public:
    /**
     * @brief Construct a new FelixBusReader object
     *
     * @param path The path to the bus files
     * @param groupname The group name for the bus files
     * @param verbose Whether to print verbose output
     * @param ignore_stale Whether to disregard stale files
     * @param timeout The timeout for acquiring a lock on the bus file
     */
    explicit FelixBusReader(std::string path = "bus",
                            std::string groupname = "FELIX",
                            bool verbose = false,
                            bool ignore_stale = true,
                            std::chrono::milliseconds timeout = 100ms) :
      m_bus_path_prefix{std::move(path)},
      m_groupname{std::move(groupname)},
      m_timeout{timeout},
      m_verbose{verbose},
      m_ignore_stale{ignore_stale}
    {}

    /**
     * @brief Get information for a FID from the bus
     *
     * @param fid The FID to get information for
     * @return FelixBusInfo The information for the FID
     * @throws FailedOpeningBusException if the bus file cannot be opened
     * @throws LockedException if the bus file cannot be locked within timeout
     * @throws FailedParsingBusException if the bus file cannot be parsed
     * @throws BusNotFoundException if the bus directory does not exist
     * @throws EntryNotFoundException if the FID is not found in the bus file
     */
    [[nodiscard]] FelixBusInfo get_info(uint64_t fid);

    /**
     * @brief Check if a file is stale
     *
     * @param path The path to the file
     * @return bool True if the file is stale
     * @throws std::filesystem::filesystem_error if the modification time cannot be retrieved
     */
    [[nodiscard]] static bool is_stale(const std::filesystem::path& path);

  private:
    /**
     * @brief Read a bus file
     *
     * @param json_file The file to read
     * @return std::optional<simdjson::dom::document_stream> The parsed content or std::nullopt
     * @throws std::system_error if the write lock cannot be acquired due to an OS error
     * @throws LockedException if the write lock cannot be acquired due to timeout
     * @throws std::filesystem::filesystem_error if file access fails
     */
    [[nodiscard]] std::optional<simdjson::dom::document_stream> read_file(
      std::filesystem::directory_entry& json_file);

    /**
     * @brief Find an entry in a bus file
     *
     * Checks for both non-stream and stream entries. May read the file twice for this.
     *
     * @param json_file The file to search in
     * @param fid The fid to search for
     * @return std::optional<FelixBusInfo> The found element or std::nullopt
     */
    [[nodiscard]] std::optional<FelixBusInfo> find_entry(
      std::filesystem::directory_entry& json_file,
      uint64_t fid);

    /**
     * @brief Search for an fid in a bus file
     *
     * @param content The content of the bus file
     * @param fid The fid to search for
     * @param check_stream Whether to check for stream entries
     * @return std::optional<FelixBusInfo> The found element or std::nullopt
     */
    [[nodiscard]] std::optional<FelixBusInfo> search_item(
      simdjson::dom::document_stream& content,
      uint64_t fid,
      bool check_stream) const;

    /**
     * @brief Search for a stream fid in a bus file
     *
     * Zeroes out stream ID and tries to match the resulting FID for an entry which has stream=true
     * (returns last matching entry)
     *
     * @param content The content of the bus file
     * @param fid The fid to search for
     * @return std::optional<FelixBusInfo> The found element or std::nullopt
     */
    [[nodiscard]] std::optional<FelixBusInfo>
    search_item_stream(simdjson::dom::document_stream& content, uint64_t fid) const;

    /**
     * @brief Parse the data from a bus file entry
     *
     * @param data The data to parse
     * @return FelixBusInfo The parsed information
     * @throws simdjson::simdjson_error if the data cannot be parsed
     */
    [[nodiscard]] static FelixBusInfo parse_data(const simdjson::dom::element& data);

    std::string m_bus_path_prefix;
    std::string m_groupname;
    std::chrono::milliseconds m_timeout;
    bool m_verbose{};
    bool m_ignore_stale{};
    simdjson::dom::parser m_parser;
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_FELIXBUS_HPP