#ifndef FELIXBUSFS_FELIXBUSWRITER_HPP
#define FELIXBUSFS_FELIXBUSWRITER_HPP

#include <chrono>
#include <filesystem>
#include <functional>
#include <set>
#include <string>
#include <unordered_map>

#include <simdjson.h>

#include "felixbus/FelixBusFile.hpp"
#include "felixbus/FelixBusInfo.hpp"

using namespace std::chrono_literals;

namespace felixbus {
  using GetCurrentInfoFunc = std::function<std::pair<std::set<FelixBusInfo>, std::string>()>;

  /**
   * @brief A class writing FELIX bus information to files
   *
   * The update_info_func is passed to the @ref FelixBusFile objects to update the bus files if
   * passed. Bus files are kept up to date by touching them periodically. If cleanup is set to true,
   * the bus files and directories will be removed on destruction. The timeout is the maximum amount
   * of time waited to acquire a lock on the bus file. If the lock cannot be acquired, the write
   * operation will throw a LockedException.
   */
  class FelixBusWriter
  {
  public:
    /**
     * @brief Construct a new FelixBusWriter object
     *
     * @param path The path to the bus files
     * @param groupname The group name for the bus files
     * @param update_info_func A function that returns the current set of FelixBusInfo
     * @param verbose Whether to print verbose output
     * @param cleanup Whether to cleanup the bus file and directories on destruction
     * @param timeout The timeout for acquiring a lock on the bus file
     */
    explicit FelixBusWriter(std::string path = "bus",
                            std::string groupname = "FELIX",
                            GetCurrentInfoFunc update_info_func = nullptr,
                            bool verbose = false,
                            bool cleanup = true,
                            std::chrono::milliseconds timeout = 100ms) :
      m_bus_path_prefix{std::move(path)},
      m_groupname{std::move(groupname)},
      m_timeout{timeout},
      m_verbose{verbose},
      m_cleanup{cleanup},
      m_get_current_info{std::move(update_info_func)}
    {}

    /**
     * @brief Write information to the bus files
     *
     * Writes into the provided information to them. The information will be split into multiple
     * files by DID and CID.
     *
     * @param info The information to write
     * @param filename The filename to write the information to
     * @throws FailedCreatingBusDirException if the bus directory cannot be created
     * @throws FailedOpeningBusException if the bus file cannot be opened
     * @throws FailedWritingBusException if the bus file cannot be written
     * @throws std::system_error if the write lock cannot be acquired due to an OS error
     * @throws LockedException if the write lock cannot be acquired due to timeout
     * @throws FailedGetMetadataException if the hostname or username cannot be retrieved
     */
    void publish(std::span<const FelixBusInfo> info, const std::string& filename);

    /**
     * @brief Write information about single link to the bus files
     *
     * Writes into the provided information to them. The information will be split into multiple
     * files by DID and CID.
     *
     * @param info The information to write
     * @param filename The filename to write the information to
     * @throws FailedCreatingBusDirException if the bus directory cannot be created
     * @throws FailedOpeningBusException if the bus file cannot be opened
     * @throws FailedWritingBusException if the bus file cannot be written
     * @throws std::system_error if the write lock cannot be acquired due to an OS error
     * @throws LockedException if the write lock cannot be acquired due to timeout
     * @throws FailedGetMetadataException if the hostname or username cannot be retrieved
     */
    void publish(const FelixBusInfo& info, const std::string& filename);

    /**
     * @brief Delete the bus files
     */
    void publish_close();

  private:
    /**
     * @brief Write a single FelixBusInfo to the bus files
     *
     * @param info The information to write
     * @param filename The filename to write the information to
     * @throws FailedCreatingBusDirException if the bus directory cannot be created
     * @throws FailedOpeningBusException if the bus file cannot be opened
     * @throws FailedWritingBusException if the bus file cannot be written
     * @throws std::system_error if the write lock cannot be acquired due to an OS error
     * @throws LockedException if the write lock cannot be acquired due to timeout
     * @throws FailedGetMetadataException if the hostname or username cannot be retrieved
     */
    void publish_single(const FelixBusInfo& info, const std::string& filename);

    /**
     * @brief Create the bus directory
     *
     * @param busdir The bus directory to create
     * @throws FailedCreatingBusDirException if the bus directory cannot be created
     */
    void static create_busdir(const std::filesystem::path& busdir);

    /**
     * @brief Filter information for a single file
     *
     * @param info The information to filter
     * @param filename The filename to filter for
     * @param path The path to filter for
     * @return std::set<FelixBusInfo> The filtered information
     */
    [[nodiscard]] std::set<FelixBusInfo> filter_info_by_path(
      const std::set<FelixBusInfo>& info,
      const std::string& filename,
      const std::filesystem::path& path) const;

    /**
     * @brief Get the current information for a file
     *
     * Invoke the provided callback and return the information for the provided file.
     *
     * @param path The path to get the information for
     * @return std::set<FelixBusInfo> The current information
     */
    [[nodiscard]] std::set<FelixBusInfo> get_current_info_for_file(
      const std::filesystem::path& path) const;

    std::string m_bus_path_prefix;
    std::string m_groupname;
    std::chrono::milliseconds m_timeout{};
    bool m_verbose{};
    bool m_cleanup{};
    std::unordered_map<std::filesystem::path, FelixBusFile> m_busfile_by_name;
    simdjson::dom::parser m_parser;
    GetCurrentInfoFunc m_get_current_info;
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_FELIXBUS_HPP