#ifndef FELIXBUSFS_FILELOCKER_HPP
#define FELIXBUSFS_FILELOCKER_HPP

#include <chrono>
#include <system_error>
#include <thread>

#include <fcntl.h>

using namespace std::chrono_literals;

namespace felixbus {
  /**
   * @brief A RAII-style file locking mechanism using POSIX file locks
   *
   * FileLocker provides mutual exclusion between processes using file-based locking.
   * It supports both shared (read) and exclusive (write) locks with timeout capabilities.
   * The lock is automatically released when the object is destroyed.
   */
  class FileLocker
  {
  public:
    /**
     * @brief Construct a new FileLocker object
     *
     * @param path The path to the file to lock
     * @throws std::system_error if the file cannot be opened
     */
    explicit FileLocker(const std::string& path) : m_fd{open(path.c_str(), O_RDWR | O_NONBLOCK)}
    {
      if (m_fd == -1) {
        throw std::system_error(errno, std::generic_category(), "Failed to open file for locking");
      }
    }

    /**
     * @brief Acquire a shared (read) lock on the file
     *
     * @param timeout The maximum time to wait for the lock
     * @return true if the lock was acquired, false if the timeout was reached
     * @throws std::system_error if the lock operation fails
     */
    [[nodiscard]] bool acquire_read_lock(const std::chrono::milliseconds& timeout) const
    {
      return acquire_lock(F_RDLCK, timeout);
    }

    /**
     * @brief Acquire a exclusive (write) lock on the file
     *
     * @param timeout The maximum time to wait for the lock
     * @return true if the lock was acquired, false if the timeout was reached
     * @throws std::system_error if the lock operation fails
     */
    [[nodiscard]] bool acquire_write_lock(const std::chrono::milliseconds& timeout) const
    {
      return acquire_lock(F_WRLCK, timeout);
    }

    [[nodiscard]] int fd() const { return m_fd; }

    /**
     * @brief Destructor
     *
     * Releases the lock and closes the file descriptor
     */
    ~FileLocker()
    {
      unlock();
      if (m_fd != -1) {
        close(m_fd);
      }
    }
    FileLocker(const FileLocker&) = delete;
    FileLocker& operator=(const FileLocker&) = delete;
    FileLocker(FileLocker&&) = delete;
    FileLocker& operator=(FileLocker&&) = delete;

    /**
     * @brief Release the lock
     */
    void unlock() const
    {
      struct flock fl = {};
      fl.l_type = F_UNLCK;
      fl.l_whence = SEEK_SET;
      fl.l_start = 0;
      fl.l_len = 0;
      fcntl(m_fd, F_SETLK, &fl);
    }

  private:
    /**
     * @brief Acquire a lock on the file
     *
     * @param lock_type The type of lock to acquire
     * @param timeout The maximum time to wait for the lock
     * @return true if the lock was acquired, false if the timeout was reached
     * @throws std::system_error if the lock operation fails
     */
    [[nodiscard]] bool acquire_lock(const short lock_type,
                                    const std::chrono::milliseconds& timeout) const
    {
      struct flock fl = {};
      fl.l_type = lock_type;
      fl.l_whence = SEEK_SET;
      fl.l_start = 0;
      fl.l_len = 0;

      const auto start = std::chrono::steady_clock::now();

      while (std::chrono::steady_clock::now() - start < timeout) {
        if (fcntl(m_fd, F_SETLK, &fl) == 0) {
          return true;
        }

        if (errno != EAGAIN && errno != EACCES) {
          throw std::system_error(errno, std::generic_category(), "Lock failed");
        }

        constexpr static auto sleep_time = 10ms;
        std::this_thread::sleep_for(sleep_time);
      }

      return false;
    }

    int m_fd{-1};
    pid_t m_pid{getpid()};
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_FILELOCKER_HPP