#ifndef NETIO3BACKEND_EVENTLOOP_ASIOEVENTLOOP_HPP
#define NETIO3BACKEND_EVENTLOOP_ASIOEVENTLOOP_HPP

#include <map>
#include <memory>

#include <boost/asio.hpp>

#include "netio3-backend/EventLoop/BaseEventLoop.hpp"
#include "netio3-backend/EventLoop/EventSignalHandle.hpp"
#include "netio3-backend/EventLoop/EventTimerHandle.hpp"

namespace netio3 {
  /**
   * @brief The AsioEventLoop class represents an event loop using Boost.Asio
   *
   * This class uses the asio io service as the event loop driver. File descriptors are registered
   * as stream descriptors with the io service.
   *
   * Although supporting arbitrary file descriptors (like for exaple libfabric ones), this event
   * loop is designed to be used with the asyncmsg backend.
   */
  class AsioEventLoop : public BaseEventLoop
  {
  public:
    explicit AsioEventLoop(std::function<void()> cb_init = nullptr);
    ~AsioEventLoop() override;
    AsioEventLoop(const AsioEventLoop& other) = delete;
    AsioEventLoop& operator=(const AsioEventLoop& other) = delete;
    AsioEventLoop(AsioEventLoop&& other) = delete;
    AsioEventLoop& operator=(AsioEventLoop&& other) = delete;

    /**
     * @brief Process the given event context
     *
     * Call the callback registered with a file descriptor if a callback was registered.
     *
     * @param evc The event context to process
     */
    static void process_event(const EventContext& evc);

    /**
     * @copydoc BaseEventLoop::register_fd
     */
    void register_fd(const EventContext& ctx) override;

    /**
     * @copydoc BaseEventLoop::remove_fd
     *
     * close_fd is ignored as all FDs are handled by ASIO.
     */
    void remove_fd(int fd, bool close_fd = false, bool wait = false) override;

    /**
     * @copydoc BaseEventLoop::run
     */
    void run() override;

    /**
     * @copydoc BaseEventLoop::run_for
     */
    void run_for(std::uint64_t sec) override;

    /**
     * @copydoc BaseEventLoop::stop
     */
    void stop() override;

    /**
     * @copydoc BaseEventLoop::is_running
     */
    [[nodiscard]] bool is_running() const override;

    /**
     * @brief Get the io service
     *
     * @return The io service
     */
    [[nodiscard]] boost::asio::io_service& get_io_service() { return m_ioService; }

  private:
    /**
     * @brief Finishes all outstanding events on a file descriptor
     *
     * Used to finish all outstanding events on a file descriptor before removing it. If the timeout
     * expires, the function returns and issues a warning.
     *
     * @param fd The file descriptor to finish events for
     * @param timeout The timeout to finish events
     */
    void finish_callbacks(int fd, std::chrono::milliseconds timeout);

    /**
     * Starts an asynchronous read operation on the specified stream descriptor
     *
     * Start handling events when file descriptors become readable.
     *
     * @param stream_descriptor The stream descriptor to read from
     * @param success_handler The handler to be called when the read operation is successful
     * @param error_handler The handler to be called when an error occurs during the read
     * operation
     */
    void start_async_read(
      const std::shared_ptr<boost::asio::posix::stream_descriptor>& stream_descriptor,
      const std::function<void()>& success_handler,
      const std::function<void(const boost::system::error_code&)>& error_handler) const;

    /**
     * @brief Stop the event loop
     *
     * This function is used to stop the event loop. It contains the actual implementation in a
     * non-virtual function.
     */
    void do_stop();

    boost::asio::io_service m_ioService;
    std::unique_ptr<boost::asio::io_service::work> m_work{nullptr};
    std::map<int, std::shared_ptr<boost::asio::posix::stream_descriptor>> m_streamDescriptors;
    std::map<int, EventContext> m_ev_contexts_by_fd;
    std::function<void()> m_cb_init;
    std::atomic_bool m_is_running = false;
    mutable std::recursive_mutex m_mutex;
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_EVENTLOOP_ASIOEVENTLOOP_HPP