#ifndef NETIO3BACKEND_EVENTLOOP_BASEEVENTLOOP_HPP
#define NETIO3BACKEND_EVENTLOOP_BASEEVENTLOOP_HPP
#include <chrono>
#include <functional>
#include <memory>
#include <mutex>

#include <thread>

#include <ers/ers.h>

#include "netio3-backend/EventLoop/EventSignalHandle.hpp"
#include "netio3-backend/EventLoop/EventTimerHandle.hpp"

ERS_DECLARE_ISSUE(netio3, FailedAddingFd, std::format("Could not add file descriptor {} to epoll. Events from this resource will be neglected. {}", fd, message), ((int) fd)((const std::string&) message))
ERS_DECLARE_ISSUE(netio3, FailedClosingFd, std::format("Failed to close FD {} due to {}", fd, message), ((int) fd)((const std::string&) message))
ERS_DECLARE_ISSUE(netio3, FailedRemovingFd, std::format("Tried to close FD {} which is not open. Doing nothing", fd), ((int) fd))
ERS_DECLARE_ISSUE(netio3, FailedUpdateFdFlags, std::format("Failed to change flags of file descriptor {}: {}", fd, message), ((int) fd)((const std::string&) message))
ERS_DECLARE_ISSUE(netio3, SkippedEvents, std::format("Might have skipped events while closing file descriptor {}.", fd), ((int) fd))

namespace netio3 {
  /**
   * @brief The EventContext struct holds fd and callbacks associated with it
   *
   * The callback is executed when the fd becomes readable, the on_closed_cb is executed when the fd
   * is removed from the event loop.
   */
  struct EventContext {
    int fd{-1};
    std::function<void(int)> cb{nullptr};
    std::function<void(int)> on_closed_cb{nullptr};
  };

  /**
   * @brief The BaseEventLoop class is an interface for event loops
   *
   * The BaseEventLoop class is an interface for event loops. It provides functionality to register
   * and remove file descriptors and to run the event loop.
   *
   * It implements functionality to create and remove signals and timers using the register_fd and
   * remove_fd functionalities provided by the derived classes.
   *
   * The event loop contains an instance that registers events. When an event is triggered, the
   * event loop will pick it up and execute the callback associated with the event. The event loop
   * is inherently single-threaded. Functions interacting with the event loop have to be thread
   * safe.
   */
  class BaseEventLoop : public std::enable_shared_from_this<BaseEventLoop>
  {
  public:
    BaseEventLoop() = default;
    virtual ~BaseEventLoop() = default;
    BaseEventLoop(const BaseEventLoop& other) = delete;
    BaseEventLoop& operator=(const BaseEventLoop& other) = delete;
    BaseEventLoop(BaseEventLoop&& other) = delete;
    BaseEventLoop& operator=(BaseEventLoop&& other) = delete;

    /**
     * @brief Registers an event with the event loop
     *
     * This function is used to register a file descriptor with the event loop.
     * The registered file descriptor will be monitored for read events.
     *
     * @param ctx The event context to register
     */
    virtual void register_fd(const EventContext& ctx) = 0;

    /**
     * @brief Removes an event from the event loop
     *
     * This function is used to remove a file descriptor from the event loop. The file descriptor
     * will no longer be monitored for events.
     *
     * If close_fd is true, the file descriptor is closed. Otherwise, it is only removed from the
     * event loop.
     *
     * @param fd The file descriptor to remove
     * @param close_fd Whether to close the file descriptor
     * @param wait Whether to wait for outstanding events to finish
     */
    virtual void remove_fd(int fd, bool close_fd = false, bool wait = false) = 0;

    /**
     * @brief Runs the event loop
     *
     * The event loop is executed in the caling thread. The function blocks until the event loop is
     * stopped.
     *
     * The on_init callback is executed before the event loop starts.
     */
    virtual void run() = 0;

    /**
     * @brief Runs the event loop for a given time
     *
     * The event loop is executed in the calling thread. The function blocks until the event loop is
     * stopped or the given time has passed.
     *
     * @param sec The time to run the event loop in seconds
     */
    virtual void run_for(std::uint64_t sec) = 0;

    /**
     * @brief Stops the event loop
     */
    virtual void stop() = 0;

    /**
     * @brief Checks if the event loop is running
     *
     * @return True if the event loop is running, false otherwise
     */
    virtual bool is_running() const = 0;

    /**
     * @brief Get the thread id of the event loop
     *
     * @return The thread id of the event loop
     */
    [[nodiscard]] std::optional<std::thread::id> get_thread_id() const
    {
      std::lock_guard lock(m_mutex);
      return m_thread_id;
    }

    /**
     * @brief Creates a signal
     *
     * A signal is a user defined event that can be fired by the user. The callback takes the
     * corresponding file descriptor as argument. If the callback does not need it, it can be
     * ignored.
     *
     * A signal not using semaphore logic will only be executed once even if it is fired multiple
     * times before the callback is executed. A signal using semaphore logic will be executed for
     * every time it is fired.
     *
     * @param cb The callback to execute when the signal is fired
     * @param useSemaphore Whether to use a semaphore for the signal
     * @return The handle to the signal
     */
    [[nodiscard]] EventSignalHandle create_signal(const std::function<void(int)>& cb,
                                                  bool useSemaphore);

    /**
     * @brief Removes a signal
     *
     * Outstanding events are guaranteed to be executed within a timeout before the signal is
     * removed. If the timeout expires while events are still outstanding, a warning is issued. The
     * event might still be handled but it is not guaranteed.
     *
     * @param handle The handle to the signal to remove
     */
    void remove_signal(const EventSignalHandle& handle);

    /**
     * @brief Creates a timer
     *
     * A timer is a user defined event that is executed periodically with a given frequency. The
     * first execution happens after the time has elapsed once (so not immediately). The callback
     * takes the corresponding file descriptor as argument. If the callback does not need it, it can
     * be ignored.
     *
     * @param cb The callback to execute when the timer expires
     * @return The handle to the timer
     */
    [[nodiscard]] EventTimerHandle create_timer(const std::function<void(int)>& cb);

    /**
     * @brief Removes a timer
     *
     * Outstanding events might be skipped.
     *
     * @param handle The handle to the timer to remove
     */
    void remove_timer(const EventTimerHandle& handle);

  protected:
    /**
     * @brief Updates the flags of a file descriptor
     *
     * Issues an error if the update fails. Does not remove old flags.
     *
     * @param fd The file descriptor to update
     * @param new_flags The new flags to set
     */
    static void update_flags(int fd, int new_flags);

    /**
     * @brief Sets the thread id of the event loop
     *
     * @param id The thread id to set
     */
    void set_thread_id(std::thread::id id)
    {
      std::lock_guard lock(m_mutex);
      m_thread_id = id;
    }

    /**
     * @brief Resets the thread id of the event loop
     */
    void reset_thread_id()
    {
      std::lock_guard lock(m_mutex);
      m_thread_id.reset();
    }

    /**
     * @brief Waits until a file descriptor is no longer readable
     *
     * Used to check if all outstanding events on signals have been handled before removing the
     * signal. If executed in the same thread as the event loop, the finish_function is executed,
     * otherwise the function waits for the file descriptor to become unreadable. If the timeout
     * expires, the function returns and issues a warning.
     *
     * @param fd The file descriptor to wait for
     * @param timeout The timeout to wait for the file descriptor
     * @param finish_function The function to execute if called on event loop thread
     */
    void wait_for_fd(int fd,
                     std::chrono::milliseconds timeout,
                     const std::function<void()>& finish_function) const;

    /**
     * @brief Checks if a file descriptor has outstanding events
     *
     * @param fd The file descriptor to check
     * @return True if the file descriptor has outstanding events, false otherwise
     */
    [[nodiscard]] static bool check_fd(int fd);

  private:
    std::optional<std::thread::id> m_thread_id{std::nullopt};
    mutable std::mutex m_mutex;
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_EVENTLOOP_BASEEVENTLOOP_HPP