#ifndef NETIO3BACKEND_NETWORKBUFFER_HPP
#define NETIO3BACKEND_NETWORKBUFFER_HPP

#include <algorithm>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <span>
#include <stdexcept>
#include <vector>

#include <sys/uio.h>

namespace netio3 {
  /**
   * @brief A buffer to store network data
   *
   * The NetworkBuffer class is a simple buffer to store network data. It provides functionality to
   * write data to the buffer and to retrieve the data stored in the buffer.
   *
   * It either initializes its own memery or gets transferred a span of memory to use. In the first
   * case it owns the memory and will free it when it is destroyed. In the second case it does not
   * own the memory and will not free it.
   */
  class NetworkBuffer
  {
  public:
    /**
     * @brief Construct a new NetworkBuffer object allocating memory itself
     *
     * @param size The size of the buffer
     */
    explicit NetworkBuffer(std::size_t size) :
      m_owned_data(size),
      m_data{m_owned_data},
      m_size{size}
    {}

    /**
     * @brief Construct a new NetworkBuffer object using the given memory
     *
     * @param data The memory to use
     */
    explicit NetworkBuffer(std::span<std::uint8_t> data) :
      m_data{data}, m_size{data.size()}
    {}

    virtual ~NetworkBuffer() = default;
    NetworkBuffer(const NetworkBuffer&) = delete;
    NetworkBuffer(NetworkBuffer&&) = default;
    NetworkBuffer& operator=(const NetworkBuffer&) = delete;
    NetworkBuffer& operator=(NetworkBuffer&&) = default;

    /**
     * @brief Write data to the buffer
     *
     * @param data The data to write
     * @throws std::runtime_error if the buffer overflows
     */
    void write(const std::span<const std::uint8_t> data)
    {
      const auto size = data.size();
#ifndef NETIO3_BACKEND_BUFFER_UNSAFE
      if (m_pos + size > m_size) {
        throw std::runtime_error("Buffer overflow");
      }
#endif
      std::ranges::copy(data, m_data.data() + m_pos);
      m_pos += size;
    }

    /**
     * @brief Write data from an iovec to the buffer
     *
     * @param data The data to write
     * @throws std::runtime_error if the buffer overflows
     */
    void write(const iovec& data)
    {
#ifndef NETIO3_BACKEND_BUFFER_UNSAFE
      if (m_pos + data.iov_len > m_size) {
        throw std::runtime_error("Buffer overflow");
      }
#endif
      std::memcpy(m_data.data() + m_pos, data.iov_base, data.iov_len);
      m_pos += data.iov_len;
    }

    /**
     * @brief Write a value to the buffer
     *
     * @param value The value to write
     * @throws std::runtime_error if the buffer overflows
     */
    void write(const std::integral auto value)
    {
      const auto size = sizeof(value);
#ifndef NETIO3_BACKEND_BUFFER_UNSAFE
      if (m_pos + size > m_size) {
        throw std::runtime_error("Buffer overflow");
      }
#endif
      std::memcpy(m_data.data() + m_pos, &value, size);
      m_pos += size;
    }

    /**
     * @brief Reset the buffer
     *
     * Does not actually clear the buffer but resets the position to the beginning.
     */
    void reset() { m_pos = 0; }

    /**
     * @brief Get the data stored in the buffer
     *
     * @return The data stored in the buffer
     */
    [[nodiscard]] std::span<const std::uint8_t> data() const { return m_data.subspan(0, m_pos); }

    /**
     * @brief Get the data stored in the buffer (mutable)
     *
     * Required for C-APIs.
     *
     * @return The data stored in the buffer
     */
    [[nodiscard]] std::span<std::uint8_t> data() { return m_data.subspan(0, m_pos); }

    /**
     * @brief Get the size of the buffer
     *
     * @return The size of the buffer
     */
    [[nodiscard]] std::size_t size() const { return m_size; }

    /**
     * @brief Get the position in the buffer
     *
     * @return The position in the buffer
     */
    [[nodiscard]] std::size_t pos() const { return m_pos; }

    /**
     * @brief Mark a buffer as being full
     *
     * @important To be used for performance measurements not for production.
     */
    void mark_as_full() { m_pos = m_size; }

    /**
     * @brief Set the position to a given point
     *
     * @param val The position to set
     * @important To be used for performance measurements not for production.
     */
    void set_pos(std::size_t val) { m_pos = val; }

    /**
     * Implements some functionality needed by the POSIX backend. Remove it once the backend is
     * deleted.
     */
    void shuffle_to_front(std::size_t current_pos, std::size_t bytes_recv) {
      std::copy(m_data.data() + current_pos, m_data.data() + bytes_recv, m_data.data());
      m_pos = bytes_recv;
    }

    /**
     * Implements some functionality needed by the POSIX backend. Remove it once the backend is
     * deleted.
     */
    void shuffle_to_front2(std::size_t current_pos, std::size_t bytes_recv) {
      if(current_pos > 0){
        std::copy(m_data.data() + current_pos, m_data.data() + bytes_recv, m_data.data());
      }
      m_pos = bytes_recv - current_pos;
    }

  private:
    std::vector<std::uint8_t> m_owned_data{};
    std::span<std::uint8_t> m_data;
    std::size_t m_size{};
    std::size_t m_pos{};
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_NETWORKBUFFER_HPP