#ifndef NETIO3BACKEND_UTILITY_THREADSAFEMAP_HPP
#define NETIO3BACKEND_UTILITY_THREADSAFEMAP_HPP

#include <map>
#include <mutex>


namespace netio3 {
  /**
   * @brief Implements a conditionally thread-safe map
   *
   * The ThreadSafeMap class implements a map that can be used in a thread-safe or non-thread-safe
   * way. It uses a std::map as the underlying data structure and a mutex that is conditionally
   * locked.
   *
   * @tparam Key The key type of the map
   * @tparam Value The value type of the map
   */
  template<typename Key, typename Value>
  class ThreadSafeMap {
    public:
      /**
       * @brief Construct a new ThreadSafeMap object
       *
       * @param safe Whether the map should be thread-safe
       */
      explicit ThreadSafeMap(const bool safe): m_safe{safe} {}

      /**
       * @brief Wrapper around std::map::try_emplace
       *
       * @param key The key to insert
       * @param args Arguments to forward to the constructor of the value
       */
      template<typename... Args>
      bool try_emplace(const Key& key, Args&&... args) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.try_emplace(key, std::forward<Args>(args)...).second;
      }

      /**
       * @brief Wrapper around std::map::try_emplace
       *
       * @param key The key to insert
       * @param args Arguments to forward to the constructor of the value
       */
      template<typename... Args>
      bool try_emplace(Key&& key, Args&&... args) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.try_emplace(std::move(key), std::forward<Args>(args)...).second;
      }

      /**
       * @brief Wrapper around std::map::erase
       *
       * @param key The key to erase
       */
      std::size_t erase(const Key& key) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.erase(key);
      }

      /**
       * @brief Wrapper around std::map::clear
       */
      void clear() {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        m_map.clear();
      }

      /**
       * @brief Wrapper around std::map::size
       *
       * @return The size of the map
       */
      [[nodiscard]] std::size_t size() const {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.size();
      }

      /**
       * @brief Wrapper around std::map::empty
       *
       * @return True if the map is empty, false otherwise
       */
      [[nodiscard]] bool empty() const {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.empty();
      }

      /**
       * @brief Wrapper around std::map::contains
       *
       * @param key The key to check for
       * @return True if the map contains the key, false otherwise
       */
      [[nodiscard]] bool contains(const Key& key) const {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return m_map.contains(key);
      }

      /**
       * @brief Use a value in the map in a (potentially) thread-safe way
       *
       * Locks the mutex for the duration of the function call if the map is thread-safe.
       *
       * @param key The key to use
       * @param func The function to apply to the value
       * @return The result of the function
       */
      [[nodiscard]] auto apply(const Key& key, const std::invocable<const Value&> auto& func) const -> decltype(func(std::declval<const Value&>())) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return func(m_map.at(key));
      }


      /**
       * @brief Use a value in the map in a (potentially) thread-safe way
       *
       * Locks the mutex for the duration of the function call if the map is thread-safe.
       *
       * @param key The key to use
       * @param func The function to apply to the value
       * @return The result of the function
       */
      [[nodiscard]] auto apply(const Key& key, const std::invocable<Value&> auto& func) -> decltype(func(std::declval<Value&>())) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        return func(m_map.at(key));
      }

      /**
       * @brief Apply function to all elements in a (potentially) thread-safe way
       *
       * Locks the mutex for the duration of the function call if the map is thread-safe.
       *
       * @param func The function to apply to the value
       * @return The result of the function
       */
      void apply_all(const std::invocable<const Value&> auto& func) const {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        for (const auto& [key, value] : m_map) {
          func(value);
        }
      }


      /**
       * @brief Apply function to all elements in a (potentially) thread-safe way
       *
       * Locks the mutex for the duration of the function call if the map is thread-safe.
       *
       * @param func The function to apply to the value
       * @return The result of the function
       */
      void apply_all(const std::invocable<Value&> auto& func) {
        std::unique_lock lock(m_mutex, std::defer_lock);
        if (m_safe) {
          lock.lock();
        }
        for (auto& [key, value] : m_map) {
          func(value);
        }
      }

    private:
      mutable std::mutex m_mutex;
      std::map<Key, Value> m_map;
      bool m_safe{};
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_UTILITY_THREADSAFEMAP_HPP