#ifndef NETIO3_BUFFERFORMATTER_HPP
#define NETIO3_BUFFERFORMATTER_HPP 

/**
 * @file BufferFormatter.hpp
 * @brief BufferFormatter class, BefferMsg struct, and FormatterStatus enum.
 *
 * This file contains the declarations of BufferFormatter class and its functions.
 * The funtions responsible for correctly formatting data to write into a buffer (NetworkBuffer).
 * and also provide a way to decode the buffer.
 *
 */

#include <functional>
#include <netio3-backend/NetworkBuffer.hpp>

namespace netio3 {

    /**
     * @brief Enum class used to check the status of the decode operations.
     */
    enum class DecoderStatus { OK, HEADER_INVALID };

    /**
     * @brief Return struct for the BufferFormatter::decode function.
     */
    struct BufferMsg {
        uint8_t status{};
        uint64_t tag{};
        std::span<const uint8_t> payload{};
        DecoderStatus decoder_status{DecoderStatus::OK};
    };

    /**
     * @brief Enum class used to check the status of the write operation.
     */
    enum class FormatterStatus { BUFFER_OK, BUFFER_READY, MESSAGE_TOO_BIG };

    /**
     * @brief Function to convert the FormatterStatus enum to a string.
     *
     * @param status The FormatterStatus enum.
     * @return The string representation of the FormatterStatus.
     */
    [[nodiscard]] std::string formatter_status_string(FormatterStatus status);

    class BufferFormatter {
    public:
        /**
         * @brief Writes data to the buffer with the specified STATUS, TAG, and DATA.
         *
         * This is the most general write function, which allows you to specify the status, tag, and
         * data. data is a vector of iovec, which is a struct containing a pointer to the data and
         * the size of the data.
         *
         * @param buf The pointer to the buffer.
         * @param status The status value.
         * @param tag The tag value, representing the ID of an e-link.
         * @param data The data to be written.
         *
         * @return FormatterStatus::MESSAGE_TOO_BIG if the message is too big,
         * FormatterStatus::BUFFER_OK otherwise.
         */
        [[nodiscard]] FormatterStatus write(NetworkBuffer* buf,
                                            uint8_t status,
                                            uint64_t tag,
                                            std::span<const iovec> data);

        /**
         * @brief Writes data to the buffer with the specified STATUS, TAG, and DATA.
         *
         * This is the most general write function, which allows you to specify the status, tag, and
         * data. data is a span of uint8_t, which is a contiguous sequence of bytes.
         *
         * @param buf The pointer to the buffer.
         * @param status The status value.
         * @param tag The tag value, representing the ID of an e-link.
         * @param data The data to be written.
         *
         * @return FormatterStatus::MESSAGE_TOO_BIG if the message is too big,
         * FormatterStatus::BUFFER_OK otherwise.
         */
        [[nodiscard]] FormatterStatus write(NetworkBuffer* buf,
                                            uint8_t status,
                                            uint64_t tag,
                                            std::span<const std::uint8_t> data);

        /**
         * @brief Writes data to the buffer with the specified TAG and DATA, status is set to 0.
         *
         * This is a simplified write function, which allows you to specify the tag and data.
         * data is a vector of iovec, which is a struct containing a pointer to the data and the
         * size of the data.
         *
         * @param buf The pointer to the buffer.
         * @param tag The tag value, representing the ID of an e-link.
         * @param data The data to be written.
         *
         * @return FormatterStatus::MESSAGE_TOO_BIG if the message is too big,
         * FormatterStatus::BUFFER_OK otherwise.
         */
        [[nodiscard]] FormatterStatus write(NetworkBuffer* buf,
                                            uint64_t tag,
                                            std::span<const iovec> data);

        /**
         * @brief Writes data to the buffer with the specified STATUS, TAG, and DATA.
         *
         * This is a simplified write function, which allows you to specify the tag and data.
         * data is a span of uint8_t, which is a contiguous sequence of bytes.
         *
         * @param buf The pointer to the buffer.
         * @param tag The tag value, representing the ID of an e-link.
         * @param data The data to be written.
         *
         * @return FormatterStatus::MESSAGE_TOO_BIG if the message is too big,
         * FormatterStatus::BUFFER_OK otherwise.
         */
        [[nodiscard]] FormatterStatus write(NetworkBuffer* buf,
                                            uint64_t tag,
                                            std::span<const std::uint8_t> data);

        /**
         * @brief Decodes the buffer and calls the callback function for each decoded message.
         *
         * @param buf The buffer to decode, which is a span of uint8_t.
         * @param cb A callback function that will be called for each decoded message.
         */
        [[gnu::visibility("default")]] static void decode(
            std::span<const std::uint8_t> buf,
            const std::function<void(std::uint64_t, std::span<const std::uint8_t>, std::uint8_t)>& cb);

        /**
         * @brief Function to be called when the buffer has been sent.
         *
         * If the buffer is reutilized, it will be added again.
         * The time complexity should remain the same as just updating the tag.
         * Set mark that no tag is present in the buffer.
         *
         * @param buf The pointer to the buffer.
         */
        void reset_buffer(NetworkBuffer* buf);

        constexpr static auto HDR_MAX_SIZE = sizeof(uint8_t) + sizeof(uint64_t) + sizeof(uint32_t);

        /**
         * @brief Fill the header of the buffer with the specified STATUS, TAG, and MESSAGE_SIZE.
         *
         * @param buf The pointer to the buffer.
         * @param status The status value.
         * @param tag The tag value, representing the ID of an e-link.
         * @param payload_size The size of the message.
         */
        void fill_header(NetworkBuffer* buf, uint8_t status, uint64_t tag, std::size_t payload_size);

        constexpr static auto HDR_STATUS_FLAG_POS = unsigned{31};
        constexpr static auto HDR_TAG_FLAG_POS = unsigned{30};
        constexpr static auto NUM_BIT_PAYLOAD_SIZE = unsigned{30};
        /*                                                                      status size tag size
         */
        constexpr static auto MSG_MAX_SIZE =
          (1U << NUM_BIT_PAYLOAD_SIZE) - sizeof(uint8_t) - sizeof(uint64_t);
    private:
        /**
         * @brief Check if the buffer has enough space to write the payload.
         *
         * Checks assuming the maximum header size of 13 bytes.
         *
         * @param buf The pointer to the buffer.
         * @param payload_size The size of the payload.
         * @return True if the buffer has enough space, false otherwise.
         */
        [[nodiscard]] bool check_size(NetworkBuffer* buf, std::size_t payload_size);

        std::uint64_t m_previous_tag{};
        bool m_buffer_empty{true};
    };

}  // namespace netio3

#endif  // NETIO3_BUFFERFORMATTER_HPP
