#ifndef NETIO3_NETIOSENDER_HPP
#define NETIO3_NETIOSENDER_HPP

#include <chrono>
#include <cstdint>
#include <cstddef>
#include <memory>
#include <set>
#include <span>
#include <unordered_map>
#include <vector>

#include <sys/uio.h>  // for iovec

#include <tbb/concurrent_hash_map.h>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>
#include <netio3-backend/EventLoop/EventSignalHandle.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3-backend/NetworkBuffer.hpp>

#include "netio3/MonitoringStats.hpp"
#include "netio3/BufferFormatter.hpp"

namespace netio3 {
    struct [[gnu::visibility("default")]] NetioSenderConfig {
        static constexpr uint64_t default_flush_interval = 1000;
        /// Which netio3 backend to use
        NetworkType backend_type{NetworkType::LIBFABRIC};
        /// Network mode for backend, passed to backend
        NetworkMode backend_mode{NetworkMode::RDMA};
        /// Size in bytes of send buffers
        size_t buffersize{};
        /// Number of send buffers to allocate
        uint32_t nbuffers{};
        /// microseconds before an incomplete buffer is automatically flushed
        uint64_t flush_interval{default_flush_interval};
        /// memory region start address for zero copy
        std::uint8_t* mr_start{nullptr};
        /// Thread safety model for the backend
        ThreadSafetyModel thread_safety{ThreadSafetyModel::SAFE};
    };

    /**
     * @brief The sender class is responsible for sending data to a receiver
     *
     * The sender has three different set of send functions:
     * - send_data: sends a block of data to a receiver copying data into a buffer
     * - buffered_send_data: sends a block of data to a receiver, but buffers the data
     * - zero_copy_send_data: sends a block of data to a receiver without copying data
     *
     * All send functions come in two variants: One taking a span of data, the other one a vector of
     * iov structs.
     */
    class [[gnu::visibility("default")]] NetioSender {
    public:
        /**
         * @brief Wrapper struct around a connection
         */
        class Connection {
            friend class NetioSender;
            NetioSender* m_sender;
            EventTimerHandle m_flush_timer;
            EndPointAddress m_ep;
            NetworkBuffer* m_current_buffer = nullptr;
            BufferFormatter m_formatter;
            bool m_ready = false;
            bool m_thread_safe{true};
            std::mutex m_mutex;

            /**
             * @brief Callback for when the flush timer expires
             *
             * Flush the buffer.
             *
             * @return NetioStatus::OK if the buffer was sent, NetioStatus::FAILED otherwise
             */
            [[nodiscard]] NetioStatus flush_callback();

        public:
            /**
             * @brief Construct a new Connection object
             *
             * @param sender Pointer to the NetioSender object
             * @param ep Remote endpoint to connect to
             * @param thread_safe Whether the connection is supposed to be thread safe
             * @param evloop Pointer to the event loop
             */
            explicit Connection(NetioSender* sender,
                                EndPointAddress ep,
                                bool thread_safe,
                                BaseEventLoop* evloop);
            Connection(const Connection&) = delete;
            Connection(Connection&&) = delete;
            Connection() = delete;
            Connection& operator=(const Connection&) = delete;
            Connection& operator=(Connection&&) = delete;
            ~Connection() = default;

            /**
             * @brief Get the endpoint of the connection
             *
             * @return The endpoint of the connection
             */
            [[nodiscard]] const EndPointAddress& get_endpoint() const { return m_ep; }
        };

        /**
         * @brief Constructor for the NetioSender
         *
         * @param config configuration of the network backend and buffering
         * @param evloop pointer to the event loop that schedules netio ops
         */
        [[gnu::visibility("default")]]
        explicit NetioSender(const NetioSenderConfig& config,
                             std::shared_ptr<BaseEventLoop> evloop);

        NetioSender(const NetioSender&) = delete;
        NetioSender(const NetioSender&&) = delete;
        NetioSender() = delete;
        NetioSender& operator=(const NetioSender&) = delete;
        NetioSender& operator=(const NetioSender&&) = delete;

        ~NetioSender() = default;

        /**
         * @brief Open a connection to a remote endpoint
         *
         * This method may return before the connection to the remote
         * endpoint is established. The user can be notified when the
         * connection has been established by setting the appropriate
         * callback.
         *
         * @param ep Remote endpoint to connect to
         * @return Reference to the connection object
         * @throws FailedOpenSendEndpoint or other exceptions from backend
         */
        [[gnu::visibility("default")]]
        Connection& open_connection(const EndPointAddress& ep);

        /**
         * @brief Close a connection to a remote endpoint
         *
         * Success will be reported by the on_connection_closed callback.
         *
         * @param ep Remote endpoint to disconnect from
         */
        [[gnu::visibility("default")]]
        void close_connection(const EndPointAddress& ep);

        /**
         * @brief Send data to a remote endpoint
         *
         * @param ep Endpoint to send the data to
         * @param tag Tag identifying the data
         * @param data Descriptor of the data to send
         * @param user_status User supplied status code to be added to header
         *
         * @retval NetioStatus::OK The data have been succefully queued for sending
         * @retval NetioStatus::NO_RESOURCES No data buffers are currently availabe, try again
         * later.
         * @retval NetioStatus::FAILED Data do not fit in send buffer or some other error occurred
         * in the backend, a retry will not work
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(const EndPointAddress& ep,
                                            uint64_t tag,
                                            std::span<const uint8_t> data,
                                            uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(const EndPointAddress& ep,
                                            uint64_t tag,
                                            std::span<const iovec> data,
                                            uint8_t user_status = 0);

        /**
         * Send to the ep given by the existing connection con.
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(Connection& con,
                                            uint64_t tag,
                                            std::span<const uint8_t> data,
                                            uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(Connection& con,
                                            uint64_t tag,
                                            std::span<const iovec> data,
                                            uint8_t user_status = 0);

        /**
         * @brief Send data to a remote endpoint
         *
         * @param ep Endpoint to send the data to
         * @param tag Tag identifying the data
         * @param data Vector of messages to be sent
         * @param user_status User supplied status code to be added to header
         *
         * Uses buffers for libfabric and copies for asynvmsg.
         *
         * @retval NetioStatus::OK The data have been succefully queued for sending
         * @retval NetioStatus::NO_RESOURCES No data buffers are currently availabe, try again
         * later.
         * @retval NetioStatus::FAILED Data do not fit in send buffer or some other error occurred
         * in the backend, a retry will not work
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(const EndPointAddress& ep,
                                            uint64_t tag,
                                            std::span<const std::span<const uint8_t>> data,
                                            uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus send_data(Connection& con,
                                            const EndPointAddress& ep,
                                            uint64_t tag,
                                            std::span<const std::span<const uint8_t>> data,
                                            uint8_t user_status = 0);

        /**
         * @brief Send data to a remote endpoint in buffered mode
         *
         * Copy the data into a network buffer. If there is insufficient space in the current
         * buffer, the current buffer will be sent and a new one allocated. The current buffer will
         * be sent when it is full or after the flush interval has elapsed.
         *
         * @param ep Endpoint to send the data to
         * @param tag Tag identifying the data
         * @param data Data to be sent
         * @param user_status User supplied status code to be added to header
         *
         * @retval NetioStatus::OK The data have been succefully queued for sending
         * @retval NetioStatus::NO_RESOURCES No data buffers are currently availabe, try again
         * later.
         * @retval NetioStatus::FAILED Data do not fit in send buffer or some other error occurred
         * in the backend, a retry will not work
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus buffered_send_data(const EndPointAddress& ep,
                                                     uint64_t tag,
                                                     std::span<const iovec> data,
                                                     uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus buffered_send_data(const EndPointAddress& ep,
                                                     uint64_t tag,
                                                     std::span<const uint8_t> data,
                                                     uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus buffered_send_data(Connection& con,
                                                     uint64_t tag,
                                                     std::span<const iovec> data,
                                                     uint8_t user_status = 0);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus buffered_send_data(Connection& con,
                                                     uint64_t tag,
                                                     std::span<const uint8_t> data,
                                                     uint8_t user_status = 0);

        /**
         * @brief Send data to a remote endpoint in zero-copy mode
         *
         * Copies the user status into a header buffer slot and sends it together
         * with the data to the provided address. The data itself is not copies and
         * and has to be inside a registered memory region.
         *
         * @param ep Endpoint to send the data to
         * @param tag Tag identifying the data
         * @param data Data to be sent
         * @param user_status User supplied status code to be added to header
         * @param key Key provided in on_send_completed callback
         *
         * @retval NetioStatus::OK The data have been succefully queued for sending
         * @retval NetioStatus::NO_RESOURCES No data buffers are currently availabe, try again
         * later.
         * @retval NetioStatus::FAILED Data do not fit in send buffer or some other error occurred
         * in the backend, a retry will not work
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus zero_copy_send_data(const EndPointAddress& ep,
                                                      std::uint64_t tag,
                                                      std::span<const iovec> iov,
                                                      uint8_t user_status,
                                                      std::uint64_t key);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus zero_copy_send_data(const EndPointAddress& ep,
                                                      std::uint64_t tag,
                                                      std::span<std::uint8_t> data,
                                                      uint8_t user_status,
                                                      std::uint64_t key);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus zero_copy_send_data(Connection& con,
                                                      std::uint64_t tag,
                                                      std::span<const iovec> iov,
                                                      uint8_t user_status,
                                                      std::uint64_t key);

        /**
         * @overload
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus zero_copy_send_data(Connection& con,
                                                      std::uint64_t tag,
                                                      std::span<std::uint8_t> data,
                                                      uint8_t user_status,
                                                      std::uint64_t key);

        /**
         * @brief Flush current buffer for given endpoint
         *
         * @param ep Endpoint whose buffer is to be flushed
         * @return NetioStatus Status of sending the buffer
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus flush_buffer(const EndPointAddress& ep);

        /**
         * @brief Get the number of available buffers for all endpoints
         *
         * @return the statistics
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] std::vector<BufferStats> get_num_available_buffers();

        /**
         * @brief Set a callback function to be called when connection is established
         *
         * @param cb Callback function to be called
         */
        [[gnu::visibility("default")]]
        void set_on_connection_established(const OnConnectionEstablishedCb& cb);

        /**
         * @brief Set a callback function to be called when connection is refused
         *
         * @param cb Callback function to be called
         */
        [[gnu::visibility("default")]]
        void set_on_connection_refused(const OnConnectionRefusedCb& cb);

        /**
         * @brief Set a callback function to be called when connection is closed
         *
         * @param cb Callback function to be called
         */
        [[gnu::visibility("default")]]
        void set_on_connection_closed(const OnConnectionClosedCb& cb);

        /**
         * @brief Set a callback function to be called when connection is closed
         *
         * Also get the keys of pending send operations for zero-copy sending. To be used by the
         * publisher and not end users.
         *
         * @param cb Callback function to be called
         */
        [[gnu::visibility("default")]]
        void set_on_connection_closed_internal(const OnConnectionClosedKeysCb& cb);

        /**
         * @brief Set a callback function to be called whenever a data send operation cmpletes
         *
         * @param cb Callback function to be called
         */
        [[gnu::visibility("default")]]
        void set_on_send_completed(const OnSendCompleted& cb);

    private:

        /**
         * @brief Check if the buffer can be written to and get a new one if not
         *
         * If no buffer is present: Try to get a new one
         * If the available space is too small: Send the buffer and try to get a new one
         * If the message is too large for a buffer: Return failed
         * If no buffers are available: Return no resources
         * Otherwise return OK
         *
         * @param con Connection object
         * @param size Size of the message
         * @return Status of the operation (see description for details)
         **/
        [[nodiscard]] NetioStatus check_buffer(Connection& con, size_t size);

        /**
         * @brief Clear bookkeeping for connection closed by backend
         *
         * @param ep Remote endpoint to disconnect from
         */
        [[gnu::visibility("hidden")]]
        void clear_connection(const EndPointAddress& ep);

        /**
         * @brief Check if a connection is ready for sending
         *
         * @throws NoConnection if the connection does not exist
         * @param ep Remote endpoint to check
         * @return true if the connection is ready, false otherwise
         */
        [[gnu::visibility("hidden")]]
        [[nodiscard]] bool check_connection(const EndPointAddress& ep);

        /**
         * @brief Create a header for a message
         *
         * @param tag Tag for the message
         * @param user_status User status for the message
         * @return NetworkBuffer containing the header
         */
        [[gnu::visibility("hidden")]]
        [[nodiscard]] NetworkBuffer create_header(std::uint64_t tag, std::uint8_t user_status, std::uint32_t payload_size);

        /**
         * @brief Callback for when a connection is closed
         *
         * Clear the connection and invoke the user callback
         *
         * @param ep Remote endpoint that was closed
         * @param keys Keys for the send operations that were in flight
         */
        [[gnu::visibility("hidden")]]
        void on_connection_closed(const EndPointAddress& ep, const std::vector<std::uint64_t>& keys);

        /**
         * @brief Callback for when a connection is established
         *
         * Mark the connection as ready and invoke the user callback
         *
         * @param ep Remote endpoint that was established
         */
        [[gnu::visibility("hidden")]]
        void on_connection_established(const EndPointAddress& ep);

        /**
         * @brief Callback for when a connection is refused
         *
         * Clear the connection and invoke the user callback
         *
         * @param ep Remote endpoint that was refused
         */
        [[gnu::visibility("hidden")]]
        void on_connection_refused(const EndPointAddress& ep);

        /**
         * @brief Callback for when a send operation completes
         *
         * Invoke the user callback
         *
         * @param ep Remote endpoint that was sent to
         * @param key Key for the send operation
         */
        [[gnu::visibility("hidden")]]
        void on_send_completed(const EndPointAddress& ep, uint64_t key);

        friend class Connection;
        std::shared_ptr<BaseEventLoop> m_evloop;
        ConnectionParameters m_conn_params;
        std::chrono::microseconds m_flush_interval;
        bool m_enable_timer;
        std::unordered_map<EndPointAddress, Connection, EndPointAddressHash> m_connections;
        std::set<EndPointAddress> m_endpoints;
        std::unique_ptr<NetworkBackend> m_backend;
        NetworkType m_backend_type{};
        bool m_thread_safe{};
        mutable std::mutex m_mutex_connection;
        mutable std::mutex m_mutex_endpoints;

        // User callback called from our connection_established callback
        OnConnectionEstablishedCb m_on_connection_established_cb = [](auto&&...) {};

        // User callback called by open_connection if open fails
        OnConnectionRefusedCb m_on_connection_refused_cb = [](auto&&...) {};

        // User callback called from our connection_closed callback.
        // NB: This is the only action in our connection closed callback which
        // would not have to exist if the user callback was passed to the
        // backend
        OnConnectionClosedCb m_on_connection_closed_cb = [](auto&&...) {};

        // User callback called from our connection_closed callback providing keys of pending send
        // operations for zero-copy sending for the publisher
        OnConnectionClosedKeysCb m_on_connection_closed_internal_cb = [](auto&&...) {};
        // User callback called from our connection_closed callback.
        // NB: This is the only action in our connection closed callback which
        // would not have to exist if the user callback was passed to the
        // backend
        OnSendCompleted m_on_send_completed_cb = [](auto&&...) {};
    };
}  // namespace netio3

#endif  // NETIO3_NETIOSENDER_HPP