#ifndef NETIO3_NETIOSUBSCRIBER_HPP
#define NETIO3_NETIOSUBSCRIBER_HPP

#include <cstdint>
#include <functional>
#include <map>
#include <memory>
#include <mutex>
#include <set>
#include <span>
#include <string>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>

#include "netio3/SubscriptionRequest.hpp"

namespace netio3
{
    class NetioSender;
    class NetioReceiver;

    // subscriber callbacks
    using CbSubscriptionConfirmed = std::function<void(uint64_t tag)>;
    using CbUnsubscriptionConfirmed = std::function<void(uint64_t tag)>;
    using CbMessageReceived = std::function<void(std::uint64_t tag, std::span<const std::uint8_t> data, std::uint8_t status)>;
    using CbBufferReceived = std::function<void(std::span<const uint8_t> buffer)>;

    /// Subscription lost callback. Will be called if a connection is
    /// closed with tags still subscribed
    using CbSubscriptionLost =
        std::function<void(const std::set<uint64_t>& tag)>;

    struct [[gnu::visibility("default")]] NetioSubscriberConfig {
        NetworkType backend_type{NetworkType::LIBFABRIC};
        NetworkMode backend_mode{NetworkMode::RDMA};
        ThreadSafetyModel thread_safety{ThreadSafetyModel::SAFE};
    };

    /**
     * @brief The subscriber class is responsible receiving data from publishers
     *
     * A subscriber contains two receivers and one sender. The sender is used to send subscription
     * requests to publishers. One of the receivers is used to receive subscription
     * acknowledgements. TCP using the ASYNCMSG provider is used for this communication. The other
     * receiver is used to receive the actual data from the publishers. The connection type is
     * determined by the config passed to the constructor.
     *
     * The subscriber also needs to know the sender mode of the publisher to know how to decode the
     * data.
     *
     * On data reception, the subscriber will call the on_data_cb callback with the received decoded
     * data and/or the on_buffer_cb callback with the received buffer.
     */
    class [[gnu::visibility("default")]] NetioSubscriber {
    public:
        /**
         * @brief Constructor for subscriber
         *
         * @param config   Configuration of the network backend and buffering
         * @param evloop   Pointer to event loop that runs netio operations
         * @param local_ep Endpoint on which we listen for incoming data
         */
        [[gnu::visibility("default")]]
        explicit NetioSubscriber(const NetioSubscriberConfig& config,
                                 std::shared_ptr<BaseEventLoop> evloop,
                                 const std::string& local_ip = "0.0.0.0");
        NetioSubscriber(const NetioSubscriber&) = delete;
        NetioSubscriber(const NetioSubscriber&&) = delete;
        NetioSubscriber() = delete;
        NetioSubscriber& operator =(const NetioSubscriber&) = delete;
        NetioSubscriber& operator =(const NetioSubscriber&&) = delete;

        /**
         * @brief Destructor for subscriber (unsubscribe all)
         */
        [[gnu::visibility("default")]]
        ~NetioSubscriber();

        /**
         * @brief Subscibe to data for a tag
         *
         * Opens connections if necessary. If this function returns NO_RESOURCES, the user should
         * try again.
         *
         * @param tag Tag for which data is required
         * @param ep Endpoint from which data are to be subscribed
         * @param conn_params  Configuration for the opened receive endpoints
         *
         * @return A NetioStatus code giving the status from the sending
         *  of the request to the endpoint (See NetioSender for details)
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus subscribe(uint64_t tag,
                                            const EndPointAddress& remote_ep,
                                            const ConnectionParametersRecv& conn_params);

        /**
         * @brief Remove subscription for a tag from an endpoint
         *
         * Closes the sender connection if no more subscriptions are active to that endpoint
         * anymore. If this function returns NO_RESOURCES, the user should try again.
         *
         * @param tag Tag for which data is to be unsubscribed
         * @param ep Endpoint from which data are to be unsubscribed
         *
         * @return A NetioStatus code giving the status from the sending
         *  of the request to the endpoint (See NetioSender for details)
         */
        [[gnu::visibility("default")]]
        [[nodiscard]] NetioStatus unsubscribe(uint64_t tag, const EndPointAddress& remote_ep);

        /**
         * @brief Set a callback for when a subscription was successful
         *
         * This means, the acknowledgement was received from the publisher.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_subscription_cb(const CbSubscriptionConfirmed& cb);

        /**
         * @brief Set a callback for when a unsubscription was successful
         *
         * This means, the acknowledgement was received from the publisher.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_unsubscription_cb(const CbUnsubscriptionConfirmed& cb);

        /** @brief Set a callback function to be called when data are received
         *
         * This callback is the main purpose of this class. Invoked for each decoded message inside
         * a buffer.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_data_cb(const CbMessageReceived& cb);

        /**
         * @brief Set a callback function to be called when data are received
         *
         * This callback is the main purpose of this class. Invoked with the entire buffer.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_buffer_cb(const CbBufferReceived& cb);

        /**
         * @brief Set a callback for when an endpoint with active subscriptions has been closed
         *
         * Callback will be called with a set of uint64_t tags which were subscribed to on the
         * endpoint that has closed.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_subscription_lost_cb(const CbSubscriptionLost& cb);

    private:
        /**
         * @brief Send an (un)subscription request to a publisher
         *
         * @param req The subscription request to send
         * @param remoteEp The endpoint to send the request to
         * @return NetioStatus code giving the status of the sending
         */
        [[nodiscard]] NetioStatus send_subscription(const SubscriptionRequest& req,
                                                    const EndPointAddress& remoteEp);

        /**
         * @brief Callback for when an acknowledgement is received
         *
         * Calls either the on_subscription or on_unsubscription callback depending on the flag.
         *
         * @param tag The tag of the subscription
         * @param flag The flag of the subscription
         */
        void on_acknowledgement(uint64_t tag, SubscriptionRequest::Sub flag);

        /**
         * @brief Callback for when a connection is lost (unexpectedly closed)
         *
         * Clean up subscriptions and invoke the on_subscription_lost callback.
         *
         * @param ep The endpoint that was lost
         */
        void connection_lost(const EndPointAddress& ep);

        /**
         * @brief Callback for when a connection is refused
         *
         * Clean up subscriptions.
         *
         * @param ep The endpoint that was refused
         */
        void connection_refused(const EndPointAddress& ep);

        CbSubscriptionConfirmed m_on_subscription = [](uint64_t) {};
        CbUnsubscriptionConfirmed m_on_unsubscription = [](uint64_t) {};
        CbMessageReceived m_on_data_cb = [](std::uint64_t, std::span<const std::uint8_t>, std::uint8_t) {};
        CbSubscriptionLost m_on_subscription_lost_cb = [](const std::set<uint64_t>&) {};

        std::unique_ptr<NetioSender> m_sender;  // For sending subscription requests
        std::unique_ptr<NetioReceiver> m_subsReceiver; // For receiving subscription acknowledgements
        std::unique_ptr<NetioReceiver> m_dataReceiver; // For receiving subscribed data

        uint16_t m_ackPort;
        std::set<uint64_t> m_subscriptions;
        std::map<uint64_t, EndPointAddress> m_endpoints;

        std::map<EndPointAddress, std::set<uint64_t>> m_connections;
        std::set<EndPointAddress> m_closing_connections;
        std::map<ConnectionParametersRecv, std::uint16_t> m_listen_ports;
        std::map<std::uint64_t, ConnectionParametersRecv> m_tag_to_conn_params;
        std::map<ConnectionParametersRecv, int> m_num_connections_per_receiver;

        std::string m_local_ip;

        bool m_thread_safe{};

        std::recursive_mutex m_mutex;
    };
} // namespace netio3

#endif // NETIO3_NETIOSUBSCRIBER_HPP
