#ifndef NETIO3_SUBSCRIPTIONENDPOINTCONTAINER_SUBSCRIPTIONENDPOINTMAP_HPP
#define NETIO3_SUBSCRIPTIONENDPOINTCONTAINER_SUBSCRIPTIONENDPOINTMAP_HPP

#include <cstdint>
#include <functional>
#include <vector>

#include <absl/container/flat_hash_map.h>

#include "netio3/NetioSender.hpp"


namespace netio3 {
  /**
   * @brief SubscriptionEndpointMap is a container for storing connections
   *
   * It uses a hash map to store the connections for each tag and supports all use cases.
   */
  class SubscriptionEndpointMap {
    public:
      /**
       * @brief Get the connections for a given tag
       *
       * @param tag The tag to get the connections for
       * @return A vector of connections for the given tag
       */
      [[nodiscard]] decltype(auto) at(std::uint64_t tag) {
        return m_map.at(tag);
      }

      /**
       * @overload
       */
      [[nodiscard]] decltype(auto) at(std::uint64_t tag) const {
        return m_map.at(tag);
      }

      /**
       * @brief Check if the map contains a given tag
       *
       * Also check that the vector is not empty.
       *
       * @param tag The tag to check for
       * @return true if the tag is present, false otherwise
       */
      [[nodiscard]] bool contains(std::uint64_t tag) const {
        return m_map.contains(tag) and not m_map.at(tag).empty();
      }

      /**
       * @brief Erase the connections for a given tag
       *
       * @param tag The tag to erase
       */
      void erase(std::uint64_t tag) {
        m_map.erase(tag);
      }

      /**
       * @brief Add a tag to the map
       *
       * @param tag The tag to add
       */
      void add(std::uint64_t tag) {
        m_map.try_emplace(tag, std::vector<std::reference_wrapper<NetioSender::Connection>>{});
      }
    private:
      absl::flat_hash_map<std::uint64_t, std::vector<std::reference_wrapper<NetioSender::Connection>>> m_map;
  };
}

#endif  // NETIO3_SUBSCRIPTIONENDPOINTCONTAINER_SUBSCRIPTIONENDPOINTMAP_HPP