#pragma once

#include <memory>

#include "felix/felix_client_thread_interface.hpp"

class FelixClientThreadExtension : public FelixClientThreadInterface {

public:
    virtual ~FelixClientThreadExtension() {};

    virtual void init_subscribe(uint64_t fid) = 0;

    /**
     * @brief Function to establish a connection before sending a message.
     * 
     * This function is used to establish a connection before sending data.
     * This function needs to be called before send_data_nb(), as send_data_nb()
     * returns whether the send operaion succeeds or not.
     * The success of this call is acknowledged by the 
     * on_connection_established() callback for the particular fid.
     *  
     * @param fids The fid for which a connection needs to be established.
     * 
     * @throws FelixClientException if the connection cannot be established.
     */
    virtual void init_send_data(uint64_t fid) = 0;

    /**
     * @brief Commands to interact with the FELIX card registers.
    */
    enum Cmd {
        UNKNOWN = 0,  ///< Unrecognized or undefined command
        NOOP = 1,     ///< No operation
        GET = 2,      ///< Read a register
        SET = 3,      ///< Write a new value in a register
        TRIGGER = 4,  ///< Not implemented
        ECR_RESET = 5 ///< Legacy command: reset the Event Counter Reset
    };

    /**
     * @brief Return code of a command.
     */
    enum Status {
        OK = 0,                     ///< Command executed successfully.
        ERROR = 1,                  ///< Generic error occurred.
        ERROR_NO_SUBSCRIPTION = 2,  ///< No subscription to receive a reply.
        ERROR_NO_CONNECTION = 3,    ///< Unable to establish connection with the target.
        ERROR_NO_REPLY = 4,         ///< No reply received within the expected time frame.
        ERROR_INVALID_CMD = 5,      ///< Command type is not recognized or supported.
        ERROR_INVALID_ARGS = 6,     ///< Arguments provided to the command are invalid or malformed.
        ERROR_INVALID_REGISTER = 7, ///< Attempted to access an invalid or unsupported register.
        ERROR_NOT_AUTHORIZED = 8    ///< Operation denied due to insufficient permissions.
    };

    struct Reply {
        uint64_t ctrl_fid;
        Status status;
        uint64_t value;
        std::string message;
    };
};
