#ifndef FELIX_DIRECT_READOUT_INTERFACE_H_
#define FELIX_DIRECT_READOUT_INTERFACE_H_

#include "felix_data_structures.hpp"

#include <cstdint>
#include <span>
#include <vector>
#include <array>

namespace felix_readout
{
    using fid_t = uint64_t;
    using elink_t = uint16_t;
    using enabled_elink_t = std::pair<fid_t, elink_t>;
    using block_spans = std::array<std::span<Block>, 2>;

    inline int get_buffer_uid(int device_id, int dma_id){
        return 10*device_id + dma_id;
    }

    inline int get_device_buffer_id(int buffer_uid){
        return buffer_uid%10;
    }

    inline int get_device_id(int buffer_uid){
        return buffer_uid/10;
    }


    class FelixInputBufferInterface
    {
        public:
            virtual ~FelixInputBufferInterface() = default;

            /**
             * @returns vector of enabled e-links for the device and DMA buffer
             * passed in the constructor. Each e-link is pair containing the
             * fid and the local e-link number.
             */ 
            [[nodiscard]] virtual std::vector<enabled_elink_t> get_enabled_elinks() = 0;

            /**
             * @returns the blocks available to read in two std::spans representing
             * the two segments of the circular DMA buffer.
             * @param processed_blocks is used to determine where to count blocks
             * from. This argument allows the support of multiple readers.
             * The function does not access contended resources.
             */
            [[nodiscard]] virtual block_spans get_available_blocks(uint64_t processed_blocks) noexcept = 0;

            /**
             * @brief set the read pointer to a new block index.
             * @details firmware will not be allowed to write over the block
             * pointed at by the index. The function is NOT thread-safe.
             */
            virtual void set_read_ptr_blocks(uint64_t processed_blocks) noexcept = 0;
    };

}


#endif /* FELIX_DIRECT_READOUT_INTERFACE_H_ */