/*
 *  ColorStream.h
 *  ers
 *
 *  Created by Jonas Roemer on 28.03.25.
 *  Copyright 2025 CERN. All rights reserved.
 *
 */

/** \file ColorStream.h This file defines ColorStream ERS stream.
  * \author Jonas Roemer
  * \brief ers header file
  */

#ifndef ERS_COLORSTREAM_HPP
#define ERS_COLORSTREAM_HPP

#include <filesystem>
#include <format>

#include <ers/OutputStream.h>
#include <ers/StandardStreamOutput.h>
#include <ers/Severity.h>

namespace ers {
  /** This class streams an issue into standard C++ output stream with colorized format.
    *
    * If running in a terminal ANSI codes are added for improved readability. If the output is
    * written to a file, the ANSI codes are removed.
    *
    * The verbosity can be controlled by the TDAQ_ERS_VERBOSITY_LEVEL environment variable:
    * * <-2: Only the message is printed
    * * -2: Message with timestamp
    * * -1: Message with timestamp and severity
    * * 0: Message with timestamp, severity, file name and line number
    * * 1: Message with timestamp, severity, file name, line number and function name
    * * >1: Message with timestamp, severity, full file name, line number and function name
    *
    * \author Jonas Roemer
    * \brief Single line, human readable, colorized format stream.
    * \tparam Device The underlying device type used for output
    */
  template<class Device>
  class ColorStream : public ers::OutputStream, public Device
  {
  public:
    using Device::device;

    /**
    * \brief Constructor
    */
    explicit ColorStream();

    /**
     * \brief Constructor
     * \param file_name The name of the file to write to
     */
    explicit ColorStream(const std::string & file_name);

    /**
    * \brief Writes the issue to the stream
    * \param issue The issue to be written
    */
    void write(const ers::Issue& issue) override;

  private:
    /**
    * \brief Returns the formatted severity of the issue as a string
    *
    * Adds color codes if the output is written to a terminal.
    *
    * \param issue The issue to be checked
    * \return The severity of the issue as a string
    */
    [[nodiscard]] std::string get_severity(const ers::Issue& issue) const;

    /**
    * \brief Returns the severity of the issue as a string
    * \param issue The issue to be checked
    * \return The severity of the issue as a string
    */
    [[nodiscard]] static std::string severity_to_string(const ers::Issue& issue);

    /**
    * \brief Returns the severity of the issue as a string with color codes
    * \param issue The issue to be checked
    * \return The severity of the issue as a string with color codes
    */
    [[nodiscard]] static std::string severity_color_code(const ers::Issue& issue);

    /**
    * \brief Checks if the output is written to a terminal
    * \param stream The output stream to be checked
    * \return True if the output is written to a terminal, false otherwise
    */
    [[nodiscard]] bool is_terminal_output(const std::ostream& stream);

    int m_verbosity_level{};
    bool m_is_terminal_output{};
  };

  template<class Device>
  ers::ColorStream<Device>::ColorStream() : m_is_terminal_output{is_terminal_output(device().stream())}, m_verbosity_level{ers::Configuration::instance().verbosity_level()} {}

  template<class Device>
  ers::ColorStream<Device>::ColorStream(const std::string & file_name) : Device(file_name), m_is_terminal_output{is_terminal_output(device().stream())}, m_verbosity_level{ers::Configuration::instance().verbosity_level()} {}

  template<class Device>
  void ers::ColorStream<Device>::write(const ers::Issue& issue)
  {
    if (m_verbosity_level < -2) {
      device().stream() << std::format(
        "{}\n",
        issue.message());
    }
    else if (m_verbosity_level == -2) {
      device().stream() << std::format(
        "[{}] {}\n",
        issue.time<std::chrono::milliseconds>("%Y-%m-%d %H:%M:%S"),
        issue.message());
    }
    else if (m_verbosity_level == -1) {
      device().stream() << std::format(
        "[{}] [{}] {}\n",
        issue.time<std::chrono::milliseconds>("%Y-%m-%d %H:%M:%S"),
        get_severity(issue),
        issue.message());
    }
    else if (m_verbosity_level == 0) {
      device().stream() << std::format(
        "[{}] [{}] [{}:{}] {}\n",
        issue.time<std::chrono::milliseconds>("%Y-%m-%d %H:%M:%S"),
        get_severity(issue),
        std::filesystem::path{issue.context().file_name()}.filename().string(),
        issue.context().line_number(),
        issue.message());
    }
    else if (m_verbosity_level == 1) {
      device().stream() << std::format(
        "[{}] [{}] [{}:{}] [{}] {}\n",
        issue.time<std::chrono::milliseconds>("%Y-%m-%d %H:%M:%S"),
        get_severity(issue),
        std::filesystem::path{issue.context().file_name()}.filename().string(),
        issue.context().line_number(),
        issue.context().function_name(),
        issue.message());
    }
    else if (m_verbosity_level > 1) {
      device().stream() << std::format(
        "[{}] [{}] [{}:{}] [{}] {}\n",
        issue.time<std::chrono::milliseconds>("%Y-%m-%d %H:%M:%S"),
        get_severity(issue),
        issue.context().file_name(),
        issue.context().line_number(),
        issue.context().function_name(),
        issue.message());
    }
    chained().write(issue);
  }

  template<class Device>
  std::string ers::ColorStream<Device>::get_severity(const ers::Issue& issue) const
  {
    if (m_is_terminal_output) {
      return std::format("{}{}\033[0m", severity_color_code(issue), severity_to_string(issue));
    }
    return severity_to_string(issue);
  }

  template<class Device>
  std::string ers::ColorStream<Device>::severity_to_string(const ers::Issue& issue)
  {
    switch (issue.severity()) {
    case ers::severity::Fatal:
      return "critical";
    case ers::severity::Error:
      return "error";
    case ers::severity::Warning:
      return "warning";
    case ers::severity::Information:
      return "info";
    case ers::severity::Log:
      return "log";
    case ers::severity::Debug:
      return "debug";
    default:
      return "unknown";
    }
  }

  template<class Device>
  std::string ers::ColorStream<Device>::severity_color_code(const ers::Issue& issue)
  {
    switch (issue.severity()) {
    case ers::severity::Fatal:
      return "\033[1;41m";
    case ers::severity::Error:
      return "\033[1;31m";
    case ers::severity::Warning:
      return "\033[1;33m";
    case ers::severity::Information:
      return "\033[1;32m";
    case ers::severity::Log:
      return "\033[1;36m";
    case ers::severity::Debug:
      return "\033[1;34m";
    default:
      return "";
    }
  }

  template<class Device>
  bool ers::ColorStream<Device>::is_terminal_output(const std::ostream& stream)
  {
    if (&stream == &std::cout) {
      return isatty(STDOUT_FILENO);
    }
    if (&stream == &std::cerr) {
      return isatty(STDERR_FILENO);
    }

    return false;
  }

}  // namespace ers

#endif
