#ifndef FELIXSERVER_FELIXSERVER_HPP
#define FELIXSERVER_FELIXSERVER_HPP

#include <memory>
#include <thread>

#include <netio3-backend/EventLoop/EventSignalHandle.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>
#include <netio3-backend/EventLoop/BaseEventLoop.hpp>

#include "felix-server/BufferedPublisher.hpp"
#include "felix-server/Receiver.hpp"
#include "felix-server/ZeroCopyPublisher.hpp"


namespace felix_server {
  enum class EventLoopType {
    asio,
    epoll,
  };

  /**
   * @brief Class for creating and managing Felix server components
   *
   * This class provides methods to create and manage publishers, receivers, timers, and signals. It
   * therefore provides building blocks to build a flexible counterpart to felix-client.
   *
   * All components created by this class are managed by the same event loop. In contrast to
   * felix-client, you do not call functions directly, but create components. This way, you can
   * create separate publishers and receivers with different settings.
   */
  class FelixServer
  {
  public:
    /**
     * @brief Constructor for the FelixServer class
     *
     * @param evloop_type The type of event loop to use (asio or epoll)
     */
    explicit FelixServer(EventLoopType evloop_type);

    FelixServer(const FelixServer&) = delete;
    FelixServer(FelixServer&&) = delete;
    FelixServer& operator=(const FelixServer&) = delete;
    FelixServer& operator=(FelixServer&&) = delete;

    /**
     * @brief Destructor for the FelixServer class
     *
     * Stops event loop
     */
    ~FelixServer();

    /**
     * @brief Create a buffered publisher
     *
     * See @ref BufferedPublisher for details.
     *
     * @param settings The settings for the publisher
     * @param tags The tags to declare for the publisher
     * @return The publisher
     */
    [[nodiscard]] BufferedPublisher create_buffered_publisher(
      const BufferedPublisher::Settings& settings,
      std::span<const std::uint64_t> tags);

    /**
     * @brief Create a zero-copy publisher
     *
     * See @ref ZeroCopyPublisher for details.
     *
     * @param settings The settings for the publisher
     * @param tags The tags to declare for the publisher
     * @return The publisher
     */
    [[nodiscard]] ZeroCopyPublisher create_zero_copy_publisher(
      const ZeroCopyPublisher::Settings& settings,
      std::span<const std::uint64_t> tags);

    /**
     * @brief Create a receiver
     *
     * See @ref Receiver for details.
     *
     * @param settings The settings for the receiver
     * @param tags The tags to declare for the receiver
     * @return The receiver
     */
    [[nodiscard]] Receiver create_receiver(const Receiver::Settings& settings,
                                           std::span<const std::uint64_t> tags);

    /**
     * @brief Create a timer
     *
     * A callback that is called on the event loop thread at a regular interval once started.
     * Destroying the handle will unregister the timer. An arbitrary number of timers can be
     * created.
     *
     * @param callback The callback to be called when the timer comes up
     * @return The timer handle
     */
    [[nodiscard]] netio3::EventTimerHandle create_timer(const EventLoopCallback& callback);

    /**
     * @brief Create a signal
     *
     * A callback that is executed on the event loop thread once after the signal is triggered.
     * With semaphore logic, the signal will be triggered once per trigger. Without semaphore, if
     * the signal was triggered multiple times before it was picked up by the event loop, it will
     * only be executed once. Destroying the handle will unregister the signal. An arbitrary number
     * of signals can be created.
     *
     * @param callback The callback to be called when the signal is triggered
     * @param semaphore Whether to use a semaphore logic or not
     * @return The signal handle
     */
    [[nodiscard]] netio3::EventSignalHandle create_signal(const EventLoopCallback& callback,
                                                          bool semaphore);

  private:
    /**
     * @brief Create the event loop
     *
     * @param evloop_type The type of event loop to create
     * @return The event loop
     */
    [[nodiscard]] std::shared_ptr<netio3::BaseEventLoop> create_event_loop(
      EventLoopType evloop_type);

    std::shared_ptr<netio3::BaseEventLoop> m_event_loop;
    std::jthread m_evloop_thread;
    mutable std::mutex m_mutex;
  };
}  // namespace felix_server

#endif  // FELIXSERVER_FELIXSERVER_HPP