#ifndef FELIXSERVER_RECEIVER_HPP
#define FELIXSERVER_RECEIVER_HPP

#include <cstddef>
#include <cstdint>
#include <string>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/NetioPublisher.hpp>
#include <netio3/NetioReceiver.hpp>
#include <felixbus/FelixBusWriter.hpp>

#include "felix-server/Definitions.hpp"

namespace felix_server {
  /**
   * @brief Class for receiving data from senders
   *
   * The receiver can be configured using the settings struct. The settings include the IP and port
   * ofnthe publisher, the network type (LIBFABRIC for RDMA or ASYNCMSG for TCP), the buffer size,
   * and number of buffers per sender.
   *
   * The parameters are written to the bus given the path, group, and filename.
   *
   * Two callbacks can be provided: One invoked per received network buffer and one invoked for each
   * decoded message.
   *
   * Note that the receiver is not the counterpart of the publisher. The receiver is used to receive
   * data from senders (e.g. felix-client sending data).
   *
   * @note The receiver is supposed to be created from the @ref FelixServer class and not directly.
   */
  class Receiver
  {
  public:
    struct Settings {
      std::string ip{};
      std::uint16_t port{};
      netio3::NetworkType network_type{};
      std::size_t buffer_size{};
      std::uint32_t buffer_count{};
      std::string bus_path{};
      std::string bus_group{};
      std::string bus_filename{};
      bool buffered{};
      OnMessageCallback on_msg{nullptr};
      OnBufferCallback on_buffer{nullptr};
    };

    /**
     * @brief Constructor for the Receiver class
     *
     * Only data for the given tags can be published (only these tags are declared in the bus and
     * clients can therefore only send data to these tags).
     *
     * @param settings The settings for the receiver
     * @param evloop The event loop to use for the receiver
     * @param tags The tags to declare for the receiver
     */
    Receiver(Settings settings,
             std::shared_ptr<netio3::BaseEventLoop> evloop,
             std::span<const std::uint64_t> tags);

  private:
    /**
     * @brief Declare the tags in the bus
     *
     * @param tags The tags to declare
     */
    void declare(std::span<const std::uint64_t> tags);

    Settings m_settings{};
    felixbus::FelixBusWriter m_bus_writer{};
    netio3::NetioReceiver m_receiver;
    std::uint16_t m_port{};
  };
}  // namespace felix_server

#endif  // FELIXSERVER_RECEIVER_HPP