#ifndef FELIXSERVER_ZEROCOPYPUBLISHER_HPP
#define FELIXSERVER_ZEROCOPYPUBLISHER_HPP

#include <cstddef>
#include <cstdint>
#include <string>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>
#include <netio3/NetioPublisher.hpp>
#include <felixbus/FelixBusWriter.hpp>

#include "felix-server/Definitions.hpp"

namespace felix_server {
  /**
   * @brief Class for publishing of data using zero-copy sending
   *
   * The publisher can be configured using the settings struct. The settings include the IP and port
   * ofnthe publisher, the network type (LIBFABRIC for RDMA or ASYNCMSG for TCP), the buffer size,
   * and location of the buffer (see below), and the number of buffers to be allocated by receivers.
   *
   * The parameters are written to the bus given the path, group, and filename.
   *
   * Thtree callbacks can be provided: One for subscription and one for unsubscription. These
   * callbacks are called when a subscription or unsubscription is received. The callbacks are
   * called with the tag and endpoint as parameters. The thrird callback is called when the publish
   * is completed.
   *
   * Zero-copy sending allows to send data without copying it to the sender's buffer. This is done
   * by registering a buffer upfront with the network stack. Only data from inside this buffer can
   * be sent. The data that are being sent have to remain untouched from the moment publish is
   * called until the completion callback is called. To match callbacks to the publish calls, a key
   * is passed to the publish function. This key is returned by the completion callback. The key can
   * be any 64bit number. For example, it could be the address of the data that is being sent.
   *
   * @note The publisher is supposed to be created from the @ref FelixServer class and not directly.
   * @note It is the user's responsibility to ensure that the data being sent is not modified until
   *       the publish is completed.
   */
  class ZeroCopyPublisher
  {
  public:
    struct Settings {
      std::string ip{};
      std::uint16_t port{};
      netio3::NetworkType network_type{};
      std::size_t buffer_size{};
      std::size_t num_buffers_receiver{};
      std::uint8_t* mr_start{nullptr};
      std::string bus_path{};
      std::string bus_group{};
      std::string bus_filename{};
      OnSubscriptionCallback on_sub{nullptr};
      OnUnsubscriptionCallback on_unsub{nullptr};
      OnPublishCompletedCallback on_publish_completed{nullptr};
    };

    /**
     * @brief Constructor for the ZeroCopyPublisher class
     *
     * Only data for the given tags can be published (only these tags are declared in the bus and
     * clients can therefore only subscribe to these tags).
     *
     * @param settings The settings for the publisher
     * @param evloop The event loop to use for the publisher
     * @param tags The tags to declare for the publisher
     */
    ZeroCopyPublisher(Settings settings,
                      std::shared_ptr<netio3::BaseEventLoop> evloop,
                      std::span<const std::uint64_t> tags);

    /**
     * @brief Publish data using a span of iovec
     *
     * The data is published to all subscribers of the given tag. An optional status byte can be set
     * (default is OK).
     *
     * The data is provided as a span of iovec (so does not need to be contiguous). The maximum
     * number of entries in the iovec might be limited by the network stack (typically around 30).
     * The data has to be inside the registered memory region (mr_start) and has to remain valid
     * until the publish is completed.
     *
     * This function returns
     * * OK if the publish was successful to all subscribers.
     * * FAILED if all send operations failed (indicating a retry will not work and no completion
     *   callback will be triggered)
     * * PARTIALLY_FAILED if at least one send operation failed but at least one send operation
     *   succeeded (a completion callback will be triggered)
     * * NO_SUBSCRIPTIONS if there are no subscribers for the given tag
     * * NO_RESOURCES if there are not enough resources to send the data. In this case the function
     *   must be called again with the same parameters except that retry is set to true. This must
     *   be done until it no longer returns NO_RESOURCES.
     *
     * @param tag The tag to publish to
     * @param data The data to publish
     * @param retry Whether this publish is a retry of a previous publish
     * @param user_status The user status to be put into the header
     * @param key The key to identify the publish operation
     * @return The status of the publish
     */
    [[nodiscard]] netio3::NetioPublisherStatus publish(std::uint64_t tag,
                                                       std::span<const iovec> data,
                                                       bool retry,
                                                       std::uint8_t user_status,
                                                       std::uint64_t key);

    /**
     * @brief Publish contiguous data
     *
     * The data is provided as a span of bytes (has to be contiguous). The data has to be inside
     * the registered memory region (mr_start) and has to remain valid until the publish is
     * completed.
     *
     * See @ref publish for more details and how to handle the return values.
     *
     * @param tag The tag to publish to
     * @param data The data to publish
     * @param retry Whether this publish is a retry of a previous publish
     * @param user_status The user status to be put into the header
     * @param key The key to identify the publish operation
     * @return The status of the publish
     */
    [[nodiscard]] netio3::NetioPublisherStatus publish(std::uint64_t tag,
                                                       std::span<std::uint8_t> data,
                                                       bool retry,
                                                       std::uint8_t user_status,
                                                       std::uint64_t key);

  private:
    /**
     * @brief Declare the tags in the bus
     *
     * @param tags The tags to declare
     */
    void declare(std::span<const std::uint64_t> tags);

    Settings m_settings{};
    felixbus::FelixBusWriter m_bus_writer{};
    netio3::NetioPublisher m_publisher;
  };
}  // namespace felix_server

#endif  // FELIXSERVER_ZEROCOPYPUBLISHER_HPP