#ifndef FELIXCLIENT_BLOCKDECODERSTATS_HPP
#define FELIXCLIENT_BLOCKDECODERSTATS_HPP

#include <atomic>
#include <cstdint>

namespace felix {
  /**
   * @brief Statistics for @ref BlockDecoder
   */
  class BlockDecoderStatistics
  {
  public:
    /**
     * @brief Increment the count of processed blocks
     */
    void increment_processed_blocks()
    {
      m_processed_blocks.fetch_add(1, std::memory_order_relaxed);
    }

    /**
     * @brief Increment the count of blocks with invalid data
     */
    void increment_invalid_data_blocks()
    {
      m_invalid_data_blocks.fetch_add(1, std::memory_order_relaxed);
      m_dropped_blocks.fetch_add(1, std::memory_order_relaxed);
    }

    /**
     * @brief Increment the count of blocks with invalid header
     */
    void increment_invalid_header_blocks()
    {
      m_invalid_header_blocks.fetch_add(1, std::memory_order_relaxed);
      m_dropped_blocks.fetch_add(1, std::memory_order_relaxed);
    }

    /**
     * @brief Increment the count of blocks with out-of-sequence data
     */
    void increment_invalid_oos_blocks() { m_oos_blocks.fetch_add(1, std::memory_order_relaxed); }

    /**
     * @brief Get the number of processed blocks
     *
     * @return Number of processed blocks
     */
    [[nodiscard]] std::uint64_t get_processed_blocks() const
    {
      return m_processed_blocks.load(std::memory_order_relaxed);
    }

    /**
     * @brief Get the number of dropped blocks
     *
     * @return Number of dropped blocks
     */
    [[nodiscard]] std::uint64_t get_dropped_blocks() const
    {
      return m_dropped_blocks.load(std::memory_order_relaxed);
    }

    /**
     * @brief Get the number of blocks with invalid data
     *
     * @return Number of blocks with invalid data
     */
    [[nodiscard]] std::uint64_t get_invalid_data_blocks() const
    {
      return m_invalid_data_blocks.load(std::memory_order_relaxed);
    }

    /**
     * @brief Get the number of blocks with invalid header
     *
     * @return Number of blocks with invalid header
     */
    [[nodiscard]] std::uint64_t get_invalid_header_blocks() const
    {
      return m_invalid_header_blocks.load(std::memory_order_relaxed);
    }

    /**
     * @brief Get the number of blocks with out-of-sequence data
     *
     * @return Number of blocks with out-of-sequence data
     */
    [[nodiscard]] std::uint64_t get_invalid_oos_blocks() const
    {
      return m_oos_blocks.load(std::memory_order_relaxed);
    }

  private:
    std::atomic_uint64_t m_processed_blocks{0};       //!< Number of processed blocks
    std::atomic_uint64_t m_dropped_blocks{0};         //!< Number of dropped blocks
    std::atomic_uint64_t m_invalid_data_blocks{0};    //!< Number of blocks with invalid data
    std::atomic_uint64_t m_invalid_header_blocks{0};  //!< Number of blocks with invalid header
    std::atomic_uint64_t m_oos_blocks{0};  //!< Number of blocks with out-of-sequence data
  };
}  // namespace felix

#endif  // FELIXCLIENT_BLOCKDECODERSTATS_HPP