#ifndef FELIXCLIENT_CHUNKDECODER_HPP
#define FELIXCLIENT_CHUNKDECODER_HPP

#include <atomic>
#include <cstdint>
#include <cstring>
#include <functional>
#include <span>

#include "felix/BlockDecoderBuffer.hpp"

namespace felix {
  /**
   * @brief Chunk decoder
   *
   * This class assembles subchunks into messages.
   */
  class ChunkDecoder
  {
  public:
    /**
     * @brief Error bits for a chunk
     */
    enum ChunkErrorBits {
      FW_TRUNC = 1,  //!< Truncated by firmware
      SW_TRUNC = 2,  //!< Truncated by software (received new subchunk while assembling another)
      FW_MALF = 4,   //!< Inconsistency in message structure
      FW_CRC = 8,    //!< CRC error detected
      SW_MALF = 16   //!< Inconsistency in message decoding in software (not used)
    };

    /**
     * @brief Subchunk types
     */
    enum class SubchunkType {
      NIL = 0,  //!< SubChunk is used to fill up a non-full block
      FIRST,    //!< First subchunk of a message
      LAST,     //!< Last subchunk of a message
      WHOLE,    //!< Subchunk contains whole message
      MIDDLE,   //!< Middle subchunk of a message (after first, before last)
      TIMEOUT,  //!< No data received for some given time
      OOB       //!< Out-of-band message. Implies no payload data
    };

    /**
     * @brief Decoded subchunk with extra metadata
     */
    struct Subchunk {
      std::uint64_t fid{};                   //!< FID of the subchunk
      std::span<const std::uint8_t> data{};  //!< Data of the subchunk
      std::uint8_t fw_flags{};               //!< Firmware error flags
      SubchunkType type{};                   //!< Type of the subchunk
    };

    /**
     * @brief Constructor
     *
     * @param fid FID of the decoder
     * @param has_streams True if the block has stream identifiers
     * @param on_data_callback Callback to call when a message is decoded
     */
    explicit ChunkDecoder(std::uint64_t fid,
                          bool has_streams,
                          std::function<void(std::uint64_t fid,
                                             std::span<const std::uint8_t> data,
                                             std::uint8_t status)> on_data_callback);

    /**
     * @brief Handle a new subchunk
     *
     * Call the appropriate handler depending on the type of the subchunk.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_subchunk(Subchunk& subchunk);

    /**
     * @brief Get the number of corrupted streams
     *
     * @return Number of chunks with corrupted streams
     */
    [[nodiscard]] std::uint64_t get_corrupted_streams() const
    {
      return m_counters_corrupted_streams.load(std::memory_order_relaxed);
    }

  private:
    /**
     * @brief Handle a new subchunk while the scratch buffer is not empty
     *
     * Set software truncation flag and call callback for the leftover data.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_error_while_not_empty(const Subchunk& subchunk);

    /**
     * @brief Handle a new middle/last subchunk while the scratch buffer is empty
     *
     * Set software truncation flag.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_error_while_empty();

    /**
     * @brief Handle a first subchunk
     *
     * Put data into scratch and update fid if it contains stream identifiers.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_first(const Subchunk& subchunk);

    /**
     * @brief Handle a last subchunk
     *
     * Append data to scratch and call callback.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_last(const Subchunk& subchunk);

    /**
     * @brief Handle a whole subchunk
     *
     * Call callback with the fid and data.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_whole(const Subchunk& subchunk);

    /**
     * @brief Handle a timeout subchunk
     *
     * Add SW truncation flag and handle chunk as first or last depending on the scratch.
     *
     * @param subchunk Subchunk to handle
     */
    void handle_timeout(Subchunk& subchunk);

    std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>
      m_on_data_callback;
    BlockDecoderBuffer m_scratch;
    std::atomic_uint64_t m_counters_corrupted_streams{0};
    bool m_has_streams{};
  };
}  // namespace felix

#endif  // FELIXCLIENT_CHUNKDECODER_HPP