#ifndef FELIXCLIENT_FELIXCLIENTUTILITY_HPP
#define FELIXCLIENT_FELIXCLIENTUTILITY_HPP

#include <cstdint>
#include <format>
#include <numeric>
#include <ranges>

namespace utility {
  /**
   * @brief Get the IP address from a network interface name
   *
   * If passed an IP address, it will return the IP address if it is found in the network
   * interfaces. If passed a network interface name it will return the matching IP address. Returns
   * an empty string if not found.
   *
   * @param key The IP address or network interface name
   * @return std::string The IP address (or empty string if not found)
   */
  [[nodiscard]] std::string get_value_from_getifaddrs(const std::string& key);

  /**
   * @brief Get the IP address from a network interface name
   *
   * Returns an empty string if not found.
   *
   * @throws felix::FelixClientException If there is a problem with getifaddrs
   * @param interface The network interface name
   * @return std::string The IP address (or empty string if not found)
   */
  [[nodiscard]] std::string get_ip_from_interface(const std::string& interface);

  /**
   * @brief Convert a range of FIDs to a string (joined by spaces)
   *
   * @tparam R A range of FIDs
   * @param fids The range of FIDs
   * @return std::string Concatenated string of FIDs
   */
  template<std::ranges::range R>
    requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
  [[nodiscard]] std::string fids_to_string(R&& fids)
  {
    return std::accumulate(
      fids.begin(), fids.end(), std::string{}, [](const std::string& acc, std::uint64_t fid) {
        return std::format("{} {:#16x}, ", acc, fid);
      });
  }
}  // namespace utility

#endif  // FELIXCLIENT_FELIXCLIENTUTILITY_HPP