#ifndef FELIXCLIENT_RECONNECTTIMER_HPP
#define FELIXCLIENT_RECONNECTTIMER_HPP

#include <algorithm>
#include <chrono>
#include <cstdint>
#include <map>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>

namespace internal {
  /**
   * @brief Timer to reconnect subscriptions and send connections
   *
   * In case of a connection loss (for example because of a server restart), connections are added
   * to this timer to re-establish the connection as soon as possible. The @ref SenderManager and
   * @ref SubscriptionManager add items to this timer when they notice a lost connection. When the
   * first item is added, the timer is started. On every tick, it will try to re-open the
   * connection. If the connection is established, the item is removed from the timer. If the
   * attempt failed it will be tried again.
   */
  class ReconnectTimer
  {
  public:
    /**
     * @brief Constructor
     *
     * @param evloop Event loop
     * @param interval Interval between reconnection attempts
     */
    ReconnectTimer(const std::shared_ptr<netio3::BaseEventLoop>& evloop,
                   std::chrono::milliseconds interval);

    /**
     * @brief Add items to the timer
     *
     * @tparam R Type of range containing the tags
     * @param callback Callback function when reconnecting
     * @param tags Tags to reconnect
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void add_items(std::function<void(std::uint64_t)> callback, const R& tags)
    {
      std::lock_guard lock{m_mutex};
      std::ranges::transform(
        tags, std::inserter(m_items, m_items.end()), [callback](std::uint64_t tag) {
          return std::make_pair(tag, callback);
        });
      std::ranges::transform(tags, std::inserter(m_state, m_state.end()), [](std::uint64_t tag) {
        return std::make_pair(tag, State::open);
      });
      check_and_start();
    }

    /**
     * @brief Mark an attempt to reconnect a tag as failed
     *
     * Causes the timer to try to reconnect the tag again.
     *
     * @param tag Tag
     */
    void reset_state(std::uint64_t tag);

    /**
     * @brief Stop the reconnection attempts for a tag
     *
     * Called when resubscription was successful.
     *
     * @param tag Tag to stop the reconnection attempts for
     */
    void remove_item(std::uint64_t tag);

    /**
     * @brief Stop the reconnection attempts for a range of tags
     *
     * Called when the connection was established.
     *
     * @tparam R Type of range containing the tags
     * @param tags Tags to stop the reconnection attempts for
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void remove_items(const R& tags)
    {
      std::lock_guard lock{m_mutex};
      for (const auto tag : tags) {
        m_items.erase(tag);
        m_state.erase(tag);
      }
      check_and_stop();
    }

    /**
     * @brief Stop the timer
     */
    void stop();

  private:
    /**
     * @brief Initiate reconnection attempts
     *
     * Subscribes/opens send connections for all pending items. Checks afterwards if no items are
     * pending anymore and stops if that is the case.
     */
    void on_timer();

    /**
     * @brief Start the timer if not running already
     */
    void check_and_start();

    /**
     * @brief Stop the timer if no more items are pending
     */
    void check_and_stop();

    enum class State {
      open,
      in_progress,
    };

    std::map<std::uint64_t, std::function<void(std::uint64_t)>> m_items;
    std::map<std::uint64_t, State> m_state;
    netio3::EventTimerHandle m_timer;
    std::chrono::milliseconds m_interval;
    mutable std::mutex m_mutex;
  };
}  // namespace internal

#endif  // FELIXCLIENT_RECONNECTTIMER_HPP