#ifndef FELIXCLIENT_RECONNECTTIMERHANDLE_HPP
#define FELIXCLIENT_RECONNECTTIMERHANDLE_HPP

#include <algorithm>
#include <functional>
#include <set>

#include "felix/ReconnectTimer.hpp"

namespace internal {

  /**
   * @brief Handle to manage tracked items in a ReconnectTimer
   *
   * When adding items to a ReconnectTimer, this handle keeps track of the added items and
   * automatically removes them from the timer when the handle is destroyed.
   */
  class ReconnectTimerHandle
  {
  public:
    /**
     * @brief Constructor
     *
     * @param reconnect_timer Reconnect timer
     */
    explicit ReconnectTimerHandle(ReconnectTimer& reconnect_timer) :
      m_reconnect_timer{reconnect_timer}
    {}

    ReconnectTimerHandle(const ReconnectTimerHandle&) = default;
    ReconnectTimerHandle& operator=(const ReconnectTimerHandle&) = default;
    ReconnectTimerHandle(ReconnectTimerHandle&&) = default;
    ReconnectTimerHandle& operator=(ReconnectTimerHandle&&) = default;

    /**
     * @brief Destructor
     *
     * Removes all tracked items from the reconnect timer
     */
    ~ReconnectTimerHandle() {
      m_reconnect_timer.get().remove_items(m_tracked_items);
    }

    /**
     * @brief Add items to the timer
     *
     * @tparam R Type of range containing the tags
     * @param callback Callback function when reconnecting
     * @param tags Tags to reconnect
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void add_items(std::function<void(std::uint64_t)> callback, const R& tags)
    {
      std::ranges::copy(tags, std::inserter(m_tracked_items, m_tracked_items.end()));
      m_reconnect_timer.get().add_items(callback, tags);
    }

    /**
     * @brief Mark an attempt to reconnect a tag as failed
     *
     * Causes the timer to try to reconnect the tag again.
     *
     * @param tag Tag
     */
    void reset_state(std::uint64_t tag) {
      m_reconnect_timer.get().reset_state(tag);
    }

    /**
     * @brief Stop the reconnection attempts for a tag
     *
     * Called when resubscription was successful.
     *
     * @param tag Tag to stop the reconnection attempts for
     */
    void remove_item(std::uint64_t tag) {
      m_tracked_items.erase(tag);
      m_reconnect_timer.get().remove_item(tag);
    }

    /**
     * @brief Stop the reconnection attempts for a range of tags
     *
     * Called when the connection was established.
     *
     * @tparam R Type of range containing the tags
     * @param tags Tags to stop the reconnection attempts for
     */
    template<std::ranges::range R>
      requires std::same_as<std::ranges::range_value_t<R>, std::uint64_t>
    void remove_items(const R& tags)
    {
      for (const auto tag : tags) {
        m_tracked_items.erase(tag);
      }
      m_reconnect_timer.get().remove_items(tags);
    }

    /**
     * @brief Stop the timer
     */
    void stop() {
      m_reconnect_timer.get().stop();
    }

  private:
    std::reference_wrapper<ReconnectTimer> m_reconnect_timer;
    std::set<std::uint64_t> m_tracked_items;
  };
}  // namespace internal

#endif  // FELIXCLIENT_RECONNECTTIMERHANDLE_HPP