#ifndef FELIXCLIENT_SENDERMANAGER_HPP
#define FELIXCLIENT_SENDERMANAGER_HPP

#include <condition_variable>
#include <memory>

#include <felixbus/FelixBusReader.hpp>
#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>

#include "felix/ReconnectTimerHandle.hpp"
#include "felix/SenderWrapper.hpp"
#include "felix/Settings.hpp"

namespace internal {
  /**
   * @brief Manager for send connections
   *
   * Opens and manages send connections as requested. A connection is requested by tag. This class
   * takes care of looking up the connection parameters for the given tag like endpoint, connection
   * type, number and size of buffers. If a new set of these parameters is discovered it opens a new
   * connection using the @ref SenderWrapper.
   *
   * Connections can be opened and closed synchronously or asynchronously. If a connection is
   * opened/closed synchronously the function either returns in the success case or throws an
   * execption if the operation failed or did not complete within the timeout. If the operation is
   * asynchronously a callback will be invoked once the operation is completed. This callback will
   * not be invoked for synchronous operations.
   *
   * Connections that are lost (closed without request) will be automatically reconnected as soon as
   * possible. The reconnection is handled by the @ref ReconnectTimer.
   */
  class SenderManager
  {
  public:
    /**
     * @brief Constructor
     *
     * @param bus_settings Bus settings
     * @param evloop Event loop
     * @param thread_safety_model The thread safety model to use for the subscribers
     * @param reconnect_timer Reconnect timer
     * @param on_connection_established Callback on connection established
     * @param on_connection_closed Callback on connection closed
     * @param on_connection_refused Callback on connection refused
     */
    SenderManager(const BusSettings& bus_settings,
                  std::shared_ptr<netio3::BaseEventLoop> evloop,
                  netio3::ThreadSafetyModel thread_safety_model,
                  ReconnectTimer& reconnect_timer,
                  std::function<void(std::uint64_t)> on_connection_established,
                  std::function<void(std::uint64_t)> on_connection_closed,
                  std::function<void(std::uint64_t)> on_connection_refused);

    /**
     * @brief Open a connection for a given tag synchronously
     *
     * If the connection does not exist yet, request to open one. If the connection was opened
     * within the timeout simply return, otherwise throw an exception.
     *
     * @throws felix::SendConnectionTimedOut if the connection was not established within the
     * timeout
     * @throws felix::SendConnectionRefused if the connection was refused
     * @throws felix::BusException if the tag is not found in the bus
     * @param tag Tag
     * @param timeout Timeout
     */
    void open_connection(std::uint64_t tag, std::chrono::milliseconds timeout);

    /**
     * @brief Open a connection for a given tag asynchronously
     *
     * If the connection does not exist yet, request to open one. Return immediately, invoke a
     * callback on completion (either on_connection_established or on_connection_refused).
     *
     * @throws felix::BusException if the tag is not found in the bus
     * @param tag Tag
     */
    void open_connection_nb(std::uint64_t tag);

    /**
     * @brief Close a connection for a given tag synchronously
     *
     * Check if this was the last tag for a network connection. If no, return immediately. If the
     * connection was closed within the timeout simply return, otherwise throw an exception.
     *
     * @throws felix::SendDisconnectionTimedOut if the connection was not closed within the timeout
     * @throws felix::BusException if the tag is not found in the bus
     * @param tag Tag
     * @param timeout Timeout
     */
    void close_connection(std::uint64_t tag, std::chrono::milliseconds timeout);

    /**
     * @brief Close a connection for a given tag asynchronously
     *
     * Check if this was the last tag for a network connection. Return immediately, invoke a
     * callback on completion (on_connection_closed).
     *
     * @throws felix::BusException if the tag is not found in the bus
     * @param tag Tag
     */
    void close_connection_nb(std::uint64_t tag);

    /**
     * @brief Close all connections synchronously
     *
     * Close all connections and wait for the operation to complete within the timeout.
     *
     * @throws felix::FailedClosingConnections if the operation did not complete within the timeout
     */
    void close_all_connections(std::chrono::milliseconds timeout);

    /**
     * @brief Close all connections asynchronously
     *
     * Close all connections and return immediately. Invoke a callback on completion (on_connection_closed).
     */
    void close_all_connections_nb();

    /**
     * @brief Check if a connection for a given tag exists
     *
     * @param tag Tag
     * @return true if the connection exists, false otherwise
     */
    [[nodiscard]] bool has_connection(std::uint64_t tag) const;

    /**
     * @brief Check if a connection for a given tag is reconnecting
     *
     * @param tag Tag
     * @return true if it is reconnecting, false otherwise
     */
    [[nodiscard]] bool is_reconnecting(std::uint64_t tag) const;

    /**
     * @brief Send data to a given tag
     *
     * A connection must have been opened before.
     *
     * @throws std::out_of_range if the tag is not found
     * @throws std::logic_error if the sender mode is unknown
     * @throws felix::SendBeforeConnected if the connection is not established
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param fid Tag
     * @param data Data to send
     * @param flush Flush the buffer
     */
    void send_data(std::uint64_t fid, std::span<const uint8_t> data, bool flush = false);

    /**
     * @brief Send data to a given tag
     *
     * Sends messages in a single transaction even if unbuffered mode is used if possible. A
     * connection must have been opened before.
     *
     * @throws std::out_of_range if the tag is not found
     * @throws std::logic_error if the sender mode is unknown
     * @throws felix::SendBeforeConnected if the connection is not established
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param fid Tag
     * @param data Data to send
     * @param flush Flush the buffer
     */
    void send_data(std::uint64_t fid, std::span<const std::span<const uint8_t>> data, bool flush = false);

    /**
     * @brief Get the number of available buffers per FID
     *
     * Returns the minimum number of buffers since the last call to this function per FID.
     *
     * @return Number of buffers per FID
     */
    [[nodiscard]] std::map<std::uint64_t, std::size_t> get_num_buffers();

  private:
    using TagConnectionMap = std::map<SenderParams,
                                      std::map<netio3::EndPointAddress, std::set<std::uint64_t>>>;
    struct ConnectionParams {
      netio3::EndPointAddress address;
      ConnectionType connection_type{ConnectionType::tcp};
      std::size_t num_pages{};
      std::size_t page_size{};
      SenderMode mode{};
    };

    /**
     * @brief Get the connection parameters for a given tag from the bus
     *
     * @throws felix::BusException if the tag is not found in the bus
     * @param fid Tag
     * @return Connection parameters
     */
    [[nodiscard]] ConnectionParams get_info_bus(std::uint64_t fid) const;

    /**
     * @brief Open a connection if necessary
     *
     * Looks up the parameters in the bus. If this connmection does not yet exist, request to open
     * one. Update internal trackers.
     *
     * @param tag Tag
     * @return true if a connection was opened, false otherwise
     */
    [[nodiscard]] bool do_open_connection(std::uint64_t tag);

    /**
     * @brief Close a connection if last tag is removed
     *
     * Update internal trackers.
     *
     * @param tag Tag
     * @return true if the connection was closed, false otherwise
     */
    [[nodiscard]] bool do_close_connection(std::uint64_t tag);

    /**
     * @brief Close a connection if last tag is removed, call callback on completion
     *
     * Makes sure that a callback is invoked when the connection was closed.
     *
     * @param tag Tag
     */
    void do_close_connection_nb(std::uint64_t tag);

    /**
     * @brief Invoke the on connection established user callback if set
     *
     * @param tag Tag
     */
    void invoke_on_connection_established_cb(std::uint64_t tag);

    /**
     * @brief Invoke the on connection closed user callback if set
     *
     * @param tag Tag
     */
    void invoke_on_connection_closed_cb(std::uint64_t tag);

    /**
     * @brief Invoke the on connection refused user callback if set
     *
     * @param tag Tag
     */
    void invoke_on_connection_refused_cb(std::uint64_t tag);

    /**
     * @brief Get the overall number of connections (tags)
     *
     * @return Number of connections
     */
    [[nodiscard]] std::size_t get_num_connections() const;

    /**
     * @brief Update internal trackers
     *
     * @param map map to update
     * @param sender_params Connection parameters
     * @param address Endpoint address
     * @param tag Tag
     */
    static void add_tag_to_tracker(TagConnectionMap& map,
                                   const SenderParams& sender_params,
                                   const netio3::EndPointAddress& address,
                                   std::uint64_t tag);

    /**
     * @brief Get the tags that are reconnecting for a given sender and address
     *
     * @param sender_params Connection parameters
     * @param address Endpoint address
     * @return Tags that are reconnecting for the given sender and address
     */
    [[nodiscard]] std::vector<std::uint64_t> get_reconnecting_tags(
      const SenderParams& sender_params,
      const netio3::EndPointAddress& address) const;

    /**
     * @brief Callback invoked by @ref SenderWrapper when connection established
     *
     * Check if the connection was on the list of timed out connections. If yes, close the
     * connection. If the list was on the list of reconnecting tags, update the tracker. If the
     * connection was opened asychronously, call the on connection established callback.
     *
     * @param sender_params Connection parameters
     * @param address Endpoint address
     */
    void on_connection_established(const SenderParams& sender_params,
                                   const netio3::EndPointAddress& address);

    /**
     * @brief Callback invoked by @ref SenderWrapper when connection refused
     *
     * If the connection was on the list of reconnecting tags, reset the state (notify it was not
     * re-established). Update the internal trackers, call the on connection refused callback if the
     * connection was opened asynchronously.
     *
     * @param sender_params Connection parameters
     * @param address Endpoint address
     */
    void on_connection_refused(const SenderParams& sender_params,
                               const netio3::EndPointAddress& address);

    /**
     * @brief Callback invoked by @ref SenderWrapper when connection closed
     *
     * Update the internal trackers. If the connection was closed asynchronously, call the on
     * connection closed callback.
     *
     * @param sender_params Connection parameters
     * @param address Endpoint address
     */
    void on_connection_closed(const SenderParams& sender_params,
                              const netio3::EndPointAddress& address);

    /**
     * @brief Callback invoked by @ref SenderWrapper when connection lost
     *
     * Update internal trackers and add the tags to the reconnect timer.
     *
     * @param sender_params Connection parameters
     * @param address Endpoint address
     */
    void on_connection_lost(const SenderParams& sender_params,
                            const netio3::EndPointAddress& address);

    mutable felixbus::FelixBusReader m_bus;
    std::shared_ptr<netio3::BaseEventLoop> m_evloop;
    netio3::ThreadSafetyModel m_thread_safety_model{};
    ReconnectTimerHandle m_reconnect_timer;
    std::map<SenderParams, SenderWrapper> m_senders;
    std::map<std::uint64_t, SenderWrapper&> m_tag_to_sender;
    std::map<std::uint64_t, ConnectionParams> m_tag_to_params;
    TagConnectionMap m_address_to_tags_per_sender;
    std::set<std::uint64_t> m_call_onopen_cb_fids;
    std::set<std::uint64_t> m_call_onclose_cb_fids;
    std::set<std::uint64_t> m_timedout_connection_fids;
    std::set<std::uint64_t> m_tags_reconnecting;
    std::function<void(std::uint64_t)> m_on_connection_established_cb;
    std::function<void(std::uint64_t)> m_on_connection_closed_cb;
    std::function<void(std::uint64_t)> m_on_connection_refused_cb;
    std::condition_variable m_cv_connect;
    std::condition_variable m_cv_close;
    mutable std::mutex m_mutex;
  };
}  // namespace internal

#endif  // FELIXCLIENT_SENDERMANAGER_HPP