#ifndef FELIXCLIENT_TRICKLECONFIGCONTAINER_HPP
#define FELIXCLIENT_TRICKLECONFIGCONTAINER_HPP

#include <functional>
#include <mutex>
#include <vector>

namespace trickle {

/**
 * @brief Container class for trickle messages with thread-safe operations
 *
 * This class provides thread-safe storage and manipulation of trickle messages.
 * It uses mutex locking to ensure safe concurrent access from multiple threads.
 */
class TrickleConfigContainer {

public:
    /**
     * @brief Internal structure representing a single trickle message
     *
     * Contains the fid identifier and payload data for a message.
     */
    struct TrickleMessage {
        uint64_t fid;           ///< FID to send the specific configuration to
        std::vector<uint8_t> data; ///< Message payload
    };

    /**
     * @brief Clears all stored messages in a thread-safe manner
     *
     * Acquires a lock on the container and removes all stored messages.
     */
    void reset() {
        std::lock_guard lock(m_mutex);
        m_messages.clear();
    }    

    /**
     * @brief Adds a new message to the container in a thread-safe manner
     *
     * @param fid The fid identifier for the message
     * @param data The binary payload of the message
     */
    void push_back(const uint64_t fid, const std::vector<uint8_t>& data) {
        std::lock_guard lock(m_mutex);
        m_messages.push_back({fid, data});
    }
    
    /**
     * @brief Applies a function to each message in a thread-safe manner
     * 
     * The function is used to compose the JSON message given a vector of TrickleMessages.
     * 
     * @param func Function to apply to each message
     */
    void for_each(const std::function<void(const TrickleMessage&)>& func) const {
        std::lock_guard lock(m_mutex);
        for (const auto& msg : m_messages) {
            func(msg);
        }
    }

private:
    std::vector<TrickleMessage> m_messages; ///< Storage for all trickle messages
    mutable std::mutex m_mutex;            ///< Mutex for thread-safe access
};

} // namespace trickle

#endif // FELIXCLIENT_TRICKLECONFIGCONTAINER_HPP