#ifndef FELIXCLIENT_FELIXCLIENT_HPP
#define FELIXCLIENT_FELIXCLIENT_HPP

#include <chrono>
#include <functional>
#include <memory>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include <tbb/concurrent_queue.h>

#include <simdjson.h>

#include <felixbus/FelixBusReader.hpp>
#include <felix/felix_client_thread_extension.hpp>
#include <felix/felix_client_thread.hpp>
#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventSignalHandle.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>

#include "felix/SenderManager.hpp"
#include "felix/Settings.hpp"
#include "felix/SubscriptionManager.hpp"
#include "felix/TrickleFormatter.hpp"

constexpr static auto FELIX_CLIENT_STATUS_OK = 0;
constexpr static auto FELIX_CLIENT_STATUS_ALREADY_DONE = 1;
constexpr static auto FELIX_CLIENT_STATUS_TIMEOUT = 2;
constexpr static auto FELIX_CLIENT_STATUS_CONNECTION_BUSY = 3;
constexpr static auto FELIX_CLIENT_STATUS_ERROR = 4;

constexpr static auto UNBUFFERED_MR_SIZE = 128 * 1024;  //128kB

using namespace std::chrono_literals;

using netio_tag_t = std::uint64_t;

/**
 * @brief The FelixClient class is the main class for interacting with FELIX
 *
 * Each felix-client instance is driven by a single event loop. The event loop is started by calling
 * run() and stopped by calling stop().
 *
 * There are two main purposes for the FelixClient class:
 * 1. Sending data to FELIX
 * 2. Receiving data from FELIX
 *
 * Additionally, functions can be executed on the event loop using the exec function or periodically
 * using the user_timer_start function (after setting the callback using callback_on_user_timer).
 *
 * Sending data:
 * For each data stream (a tag), the user must call init_send_data to initialize the data stream.
 * After that, data can be sent using the send_data function. init_send_data will check if a new
 * connection to FELIX is needed and will open a new connection if necessary. The information is
 * looked up in the bus. send_data either sends a single message or a vector of messages.
 *
 * Receiving data:
 * Data can be received by subscribing to a tag using the subscribe function. The function will
 * check if a connection needs to be established and will open a new connection if necessary. There
 * is no init_send_data equivalent for subscriptions. The data is received in the callback function
 * that is set using callback_on_data or callback_on_buffer.
 *
 * Asynchronous vs synchronous:
 * For most functions, there are two versions: one that is blocking and one that is non-blocking.
 * Non-blocking functions are denoted by a _nb suffix. All other functions are blocking. Blocking
 * functions take a timeout in which the operation must complete. If the operation does not complete
 * an exception is thrown. Check the documentation for each function for more information. If they
 * do not throw the operation was successful. Non-blocking functions do not wait for completion.
 * Instead, the result is communicated through the callback functions:
 * - callback_on_connect: called when a send connection or a subscription is established
 * - callback_on_disconnect: called when a send connection is closed or after an unsubscription
 * - callback_on_refused: called when a send connection is refused
 * - callback_on_subscriptions_lost: called when a subscription is lost (because the server went
 * down)
 * - callback_on_resubscription: called when a subscription is re-established
 * In general, synchronous functions are easier to use and should be preferred if possible.
 *
 * Further callbacks:
 * - callback_on_init: called when the event loop is initialized and the class is ready to be used
 * - callback_on_user_timer: called when the user timer is triggered
 * - callback_on_data: called when data is received for each message
 * - callback_on_buffer: called when a buffer is received
 *
 * If the connection to the server is lost all open send connections and subscriptions will
 * periodically attempt to reconnect automatically.
 */
class FelixClient {

public:
    /**
    * @brief Construct a new FelixClient object
    *
    * The address is either an IP (v4) address or the name of a network interface (e.g. vlan109).
    * The settings object contain information about the location of the bus and the event loop type.
    * Generally, choose EventLoopType::netio3_native for RDMA and EventLoopType::netio3_asio for TCP.
    *
    * @param local_ip_address The local IP address of the machine or name of the network interface
    * @param settings The settings for the FelixClient
    */
    explicit FelixClient(const std::string& local_ip_address, const FelixClientSettings& settings);

    // returns true just before on_init is called (class is ready to be used)
    /**
    * @brief Check if the event loop is running
    *
    * This function returns true just before the on_init callback is called. This means that the
    * object is ready to be used.
    *
    * @return true if the event loop is running
    * @return false if the event loop is not running
    */
    [[nodiscard]] bool is_ready() const { return m_ready; }

    // event loop control
    /**
    * @brief Start the event loop
    */
    void run();

    /**
    * @brief Stop the event loop
    *
    * @important The event loop cannot be restarted after it has been stopped.
    */
    void stop();

    // send messages to FELIX
    [[deprecated("Use send_data(netio_tag_t, std::chrono::milliseconds) instead")]]
    void init_send_data(netio_tag_t fid);

    /**
    * @brief Initialize a send connection to FELIX
    *
    * This function checks if a connection has to be established and opens a new connection if
    * needed. The function will block until the connection is established or the timeout is reached.
    * If the function returns without throwing an exception, the connection is established.
    *
    * @throws felix::SendConnectionTimedOut if the connection was not established within the
    * timeout
    * @throws felix::SendConnectionRefused if the connection was refused
    * @throws felix::BusException if the tag is not found in the bus
    * @param fid The tag to send data to
    * @param timeout The timeout for establishing the connection
    */
    void init_send_data(netio_tag_t fid, std::chrono::milliseconds timeout);

    /**
    * @brief Asynchronously initialize a send connection to FELIX
    *
    * This function checks if a connection has to be established and opens a new connection if
    * needed. The function will return immediately. The result is communicated through the on_connect
    * callback. If the connection is refused, the on_refused callback is called. If the connection
    * request already fails, an exception is thrown.
    *
    * @throws felix::BusException if the tag is not found in the bus
    * @param fid The tag to send data to
    */
    void init_send_data_nb(netio_tag_t fid);

    /**
    * @brief Send data to FELIX
    *
    * This function sends a single message to FELIX. If a buffered sender is used (depending on
    * information in the bus), the flush parameter can be used to flush the buffer and send the
    * message immediately.
    *
    * @note Can be called without init_send_data but this behavior will change and throw an
    * exception in the future.
    * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
    * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
    * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
    * @param fid The tag to send data to
    * @param data The data to send
    * @param flush Flush the buffer
    */
    void send_data(netio_tag_t fid, std::span<const uint8_t> data, bool flush = false);
    [[deprecated("Use send_data(netio_tag_t, std::span<const uint8_t>, bool) instead")]]
    void send_data(netio_tag_t fid, const uint8_t* data, size_t size, bool flush = false);
    [[deprecated("Was never implemented")]] void send_data(netio_tag_t fid,
                                                          struct iovec* iov,
                                                          unsigned n,
                                                          bool flush = false);

    /**
     * @brief Send data to FELIX
     *
     * This function sends multiple messages to FELIX. If a buffered sender is used (depending on
     * information in the bus) it is the same as calling send_data for each message. For unbuffered
     * sending, messages will still be send together in one transaction (or as slittle as possible
     * to fit into buffers).
     *
     * @note Can be called without init_send_data but this behavior will change and throw an
     * exception in the future.
     * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
     * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
     * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
     * @param fid The tag to send data to
     * @param data The data to send
     * @param flush Flush the buffer
     */
    void send_data(netio_tag_t fid, const std::vector<std::span<const uint8_t>>& msgs);
    [[deprecated(
      "Use send_data(netio_tag_t, const std::vector<std::span<const uint8_t>>&) instead")]] void
    send_data(netio_tag_t fid,
              const std::vector<const uint8_t*>& msgs,
              const std::vector<size_t>& sizes);

    [[deprecated("Use send_data instead")]] void send_data_nb(netio_tag_t fid,
                                                              const uint8_t* data,
                                                              size_t size,
                                                              bool flush = false);
    [[deprecated("Use send_data instead")]] void send_data_nb(netio_tag_t fid,
                                                              const std::vector<const uint8_t*>& msgs,
                                                              const std::vector<size_t>& sizes);

    // subscription control
    [[deprecated("init_subscribe no longer needed")]] void init_subscribe(netio_tag_t fid);

    /**
     * @brief Synchronously subscribe to a tag
     *
     * This function subscribes to a tag. The function will block until the link is subscribed
     * or the timeout is reached. If the function returns without throwing an exception, the
     * connection is established. In case the subscription fails the exception will contain more
     * information about the cause.
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @param fid The tag to subscribe to
     * @param timeout The timeout for establishing the connection
     */
    void subscribe(netio_tag_t fid, std::chrono::milliseconds timeout);

    /**
     * @brief Asynchronously subscribe to a tag
     *
     * This function subscribes to a tag. The function will return immediately. A success is
     * communicated through the on_connect callback. If the subscription request fails, an exception
     * is thrown. Other failures are not communicated (but the on_connect callback will not be
     * called).
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fid The tag to subscribe to
     * @param timeout The timeout for establishing the connection
     */
    void subscribe_nb(netio_tag_t fid);

    /**
     * @brief Synchronously subscribe to a set of tags
     *
     * This function subscribes to a set of tags. The function will block until the links are
     * subscribed or the timeout is reached. If the function returns without throwing an exception,
     * the connection is established. In case the subscription fails the exception will contain more
     * information about the cause.
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fids The tags to subscribe to
     * @param timeout The timeout for establishing the connection
     */
    void subscribe(const std::vector<netio_tag_t>& fids, std::chrono::milliseconds timeout);

    /**
     * @brief Asynchronously subscribe to a set of tags
     *
     * This function subscribes to a set of tags. The function will return immediately. A success is
     * communicated through the on_connect callback for each tag. If the subscription request fails,
     * an exception is thrown. Other failures are not communicated (but the on_connect callback will
     * not be called).
     *
     * @throws felix::BusException If the information is not available in the bus
     * @throws felix::SubscriptionFailedError If the subscription fails
     * @throws felix::SubscriptionAlreadyDoneError If all links are already subscribed
     * @param fids The tags to subscribe to
     * @param timeout The timeout for establishing the connection
     */
    void subscribe_nb(const std::vector<netio_tag_t>& fids);
    [[deprecated("Use subscribe with std::chrono::milliseconds timeout or subscribe_nb instead")]]
    int subscribe(netio_tag_t fid, uint timeoutms = 0, bool for_register = false);
    [[deprecated("Use subscribe with std::chrono::milliseconds timeout or subscribe_nb instead")]]
    int subscribe(const std::vector<netio_tag_t>& fids, uint timeoutms = 0, bool for_register = false);
    [[deprecated("Use subscribe with std::chrono::milliseconds timeout or subscribe_nb "
                "instead")]]
    [[nodiscard]] int unsubscribe(netio_tag_t fid);

    /**
     * @brief Synchronously unsubscribe from a tag
     *
     * This function unsubscribes from a tag. The function will block until the link is unsubscribed
     * or the timeout is reached. If the function returns without throwing an exception, the
     * link is unsubscribed. In case the unsubscription fails the exception will contain more
     * information about the cause.
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @throws felix::UnsubscriptionFailedError If the unsubscription fails
     * @param fid The tag to unsubscribe from
     * @param timeout The timeout for unsubscribing
     */
    void unsubscribe(netio_tag_t fid, std::chrono::milliseconds timeout);

    /**
     * @brief Asynchronously unsubscribe from a tag
     *
     * This function unsubscribes from a tag. The function will return immediately. A success is
     * communicated through the on_disconnect callback. If the unsubscription request fails, an
     * exception is thrown. Other failures are not communicated (but the on_disconnect callback will
     * not be called).
     *
     * @throws felix::UnsubscriptionError If the FID is not managed
     * @throws felix::AsyncUnsubscriptionFailedError If the unsubscription fails
     * @param fid The tag to unsubscribe from
     */
    void unsubscribe_nb(netio_tag_t fid);

    // create and manage a timer
    [[deprecated("Use user_timer_start(std::chrono::milliseconds) instead")]]
    void user_timer_start(unsigned long interval);

    /**
    * @brief Start a user timer function in the eventloop thread
    *
    * This function allows to start a user timer function in the event loop thread. The function will
    * be called periodically with the given interval. The function is set using callback_on_user_timer.
    *
    * @param interval interval time in ms
    */
    void user_timer_start(std::chrono::milliseconds interval);

    /**
    * @brief Stop the user timer function in the eventloop thread
    */
    void user_timer_stop();

    // execute on eventloop
    using UserFunction = std::function<void()>;

    /**
    * @brief Execute a user function in the eventloop thread
    *
    * @param user_function The function to execute
    */
    void exec(const UserFunction& user_function);

    // handle commands
    /**
     * @brief Send a command to felix-register
     *
     * Sends a command to get or set a value in felix-register. The function will open a connection,
     * send the command and wait for the reply. The function will block until the reply is received
     * or the timeout expired.
     *
     * @example const auto status = send_cmd({0x1000000000080000},
     * FelixClientThreadExtension::Cmd::GET, {"REG_MAP_VERSION"}, replies);
     *
     * @param fids The tags to send the command to
     * @param cmd The command to send (@see FelixClientThreadExtension::Cmd)
     * @param cmd_args The arguments for the command (e.g. register names)
     * @param[out] replies The replies from the command
     * @return The status of the command
     */
    FelixClientThreadExtension::Status send_cmd(
      const std::vector<uint64_t>& fids,
      FelixClientThreadExtension::Cmd cmd,
      const std::vector<std::string>& cmd_args,
      std::vector<FelixClientThreadExtension::Reply>& replies);

    /**
     * @brief Get the number of available buffers of all senders per FID
     *
     * Returns the minimum number of buffers since the last call to this function per FID.
     *
     * @return Number of buffers per FID for all senders
     */
    [[nodiscard]] std::map<std::uint64_t, std::size_t> get_available_send_buffers();

    /**
     * @brief Get the statistics of the block message decoder
     *
     * This function returns the statistics of all registered block decoders.
     *
     * @return Statistics of all registered block decoders
     */
    [[nodiscard]] std::map<std::uint64_t, FelixClientThread::BlockDecoderStats> get_block_decoder_stats() const;

    // callback registration
    using OnInitCallback = std::function<void()>;
    using OnDataCallback =
      std::function<void(std::uint64_t fid, std::span<const std::uint8_t> data, std::uint8_t status)>;
    using OnDataCallbackOld = std::function<
      void(std::uint64_t fid, const std::uint8_t* data, std::size_t size, std::uint8_t status)>;
    using OnBufferCallback = std::function<void(std::span<const std::uint8_t>)>;
    using OnConnectCallback = std::function<void(std::uint64_t)>;
    using OnConnectionRefusedCallback = std::function<void(std::uint64_t)>;
    using OnDisconnectCallback = std::function<void(std::uint64_t)>;
    using OnSubscriptionsLostCallback = std::function<void(const std::set<std::uint64_t>& fids)>;
    using OnResubscriptionCallback = std::function<void(std::uint64_t fid)>;
    using OnUserTimerCallback = std::function<void()>;

    /**
     * @brief Set the callback for the on_init event
     *
     * This function sets the callback that is called when the event loop is initialized.
     *
     * @param on_init The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_init(OnInitCallback on_init);

    /**
     * @brief Set the callback for the on_connect event
     *
     * This function sets the callback that is called when a send connection is established or a
     * link was subscribed. It is only invoked for asynchronously initiated requests.
     *
     * @param on_connect The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_connect(OnConnectCallback on_connect);

    /**
     * @brief Set the callback for the on_disconnect event
     *
     * This function sets the callback that is called when a send connection is closed or a link was
     * unsubscribed. It is only invoked for asynchronously initiated requests.
     *
     * @param on_disconnect The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_disconnect(OnDisconnectCallback on_disconnect);

    /**
     * @brief Set the callback for the on_refused event
     *
     * This function sets the callback that is called when a send connection is refused. It is only
     * invoked for asynchronously initiated requests.
     *
     * @param on_refused The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_refused(OnConnectionRefusedCallback on_refused);

    /**
     * @brief Set the callback for the on_subscriptions_lost event
     *
     * This function sets the callback that is called when a subscription is lost. This can happen
     * if the server goes down.
     *
     * @param on_subscriptions_lost The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_subscriptions_lost(OnSubscriptionsLostCallback on_subscriptions_lost);

    /**
     * @brief Set the callback for the on_resubscription event
     *
     * This function sets the callback that is called when a subscription is re-established.
     *
     * @param on_resubscription The callback function
     */
    [[deprecated("Set using constructor")]] void callback_on_resubscription(OnResubscriptionCallback on_resubscription);

    /**
     * @brief Set the callback for the on_user_timer event
     *
     * This function sets the callback that is called when the user timer is triggered.
     *
     * @param on_user_timer_cb The callback function
     */
    void callback_on_user_timer(OnUserTimerCallback on_user_timer_cb);

    /**
     * @brief Set the thread affinity for the event loop
     *
     * This function sets the thread affinity for the event loop. The affinity is a string that
     * describes the CPU cores that the event loop should run on. The string can contain single
     * cores (e.g. "0") or ranges (e.g. "0-3").
     *
     * @important This function must be called before run() is called.
     *
     * @param affinity The affinity string
     */
    void set_thread_affinity(const std::string& affinity);


    // Trickle configuration
    /**
     * @brief Creates a new trickle configuration
     * 
     * Initializes a new trickle configuration, clearing any previous configuration.
     */
     void create_config();

     /**
      * @brief Adds data to the current trickle configuration
      * 
      * @throws felix::TrickleLogicException when used before create_config()
      *
      * @param elink The fid of the FE device
      * @param payload The binary payload data
      */
     void append_config(const std::uint64_t fid, const std::vector<uint8_t>& payload);
 
     /**
      * @brief Sends the current trickle configuration
      * 
      * Formats and sends the current trickle configuration to FELIX.
      *
      * @note Can be called without init_send_data but this behavior will change and throw an
      * exception in the future.
      * @throws felix::TrickleLogicException when used before append_config()
      * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
      * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
      * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)

      * @param fid Where to send the configuration
      */
     void send_config(const std::uint64_t fid);
 
     /**
      * @brief Starts the trickle service
      * 
      * Sends a command to start the trickle service.
      *
      * @note Can be called without init_send_data but this behavior will change and throw an
      * exception in the future.
      * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
      * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
      * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
      *
      * @param fid FID of the device to start the trickle service on
      */
     void start_trickle(const std::uint64_t fid);
 
     /**
      * @brief Stops the trickle service
      * 
      * Sends a command to stop the trickle service.
      *
      * @note Can be called without init_send_data but this behavior will change and throw an
      * exception in the future.
      * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
      * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
      * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
      *
      * @param fid FID of the device to stop the trickle service on
      */
     void stop_trickle(const std::uint64_t fid);
 
     /**
      * @brief Selects the BCID range for the trickle configuration
      * 
      * @note Can be called without init_send_data but this behavior will change and throw an
      * exception in the future.
      * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
      * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
      * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
      *
      * @param fid The FID of the device
      * @param first_bcid The first BCID in the range
      * @param last_bcid The last BCID in the range
      */
     void select_bcids(const std::uint64_t fid, uint32_t first_bcid, uint32_t last_bcid);

     /**
      * @brief Selects the BCID range for the trickle configuration
      * 
      * @note Can be called without init_send_data but this behavior will change and throw an
      * exception in the future.
      * @throws felix::SendDuringReconnect if the connection was lost and is being re-established
      * @throws felix::ResourceNotAvailableException if NO_RESOURCES was returned (try again)
      * @throws felix::MessageTooBigException if FAILED was returned (retrying will fail again)
      *
      * @param fid The FID of the device
      * @param throttle_factor slow the trickling by N times [1 = no throttle]
      */
      void throttle_trickle(std::uint64_t fid, std::uint32_t throttle_factor);

private:
    /**
     * @brief Parse a CPU range string
     *
     * This function parses a CPU range string and returns a vector of CPU cores. Used by
     * set_thread_affinity.
     *
     * @param range The CPU range string (e.g. "0-3", or "2")
     * @return The vector of CPU cores
     */
    [[nodiscard]] static std::vector<int> parseCpuRange(const std::string &range);

    /**
     * @brief Create an event loop
     *
     * This function creates an event loop based on the event loop type.
     *
     * @throws std::logic_error If the event loop type is unknown
     * @param eventloop_type The event loop type
     * @return The event loop
     */
    [[nodiscard]] std::shared_ptr<netio3::BaseEventLoop> create_eventloop(EventLoopType eventloop_type);

    /**
     * @brief Check if we can send data to a connection
     *
     * Checks if a connection is up. Throws if we are reconnecting, opens a connection if we are not
     * connected.
     *
     * @throws felix::SendDuringReconnect If the connection is being re-established
     * @param tag The tag to check
     */
    void check_connection_for_send(std::uint64_t tag);

    // generic callbacks
    /**
     * @brief Callback for when a connection is established
     *
     * Calls callback set by callback_on_connect.
     *
     * @param tag The tag of the connection
     */
    void on_connection_established(std::uint64_t tag);

    /**
     * @brief Callback for when a connection is closed
     *
     * Calls callback set by callback_on_disconnect.
     *
     * @param tag The tag of the connection
     */
    void on_connection_closed(std::uint64_t tag);

    /**
     * @brief Callback for when a connection is refused
     *
     * Calls callback set by callback_on_refused.
     *
     * @param tag The tag of the connection
     */
    void on_connection_refused(std::uint64_t tag);

    /**
     * @brief Callback for when a subscription is lost
     *
     * Calls callback set by callback_on_subscriptions_lost.
     *
     * @param fids The tags of the lost subscriptions
     */
    void on_subscriptions_lost(const std::set<std::uint64_t>& fids);

    /**
     * @brief Callback for when a subscription is re-established
     *
     * Calls callback set by callback_on_resubscription.
     *
     * @param fid The tag of the re-established subscription
     */
    void on_resubscription(std::uint64_t fid);

    /**
     * @brief Callback for when a register message is received
     *
     * Implemented internally and not user facing.
     *
     * @param tag The tag of the connection
     * @param data The data received
     * @param status The status of the data
     */
    void on_register_msg_received(std::uint64_t tag, std::span<const std::uint8_t> data, std::uint8_t status);

    /**
     * @brief Callback for when the event loop is initialized
     *
     * Calls callback set by callback_on_init.
     */
    void on_init_eventloop();

    /**
     * @brief Callback for when the user timer is triggered
     *
     * Calls callback set by callback_on_user_timer.
     */
    void on_user_timer() const;

    /**
     * @brief Callback for when exec signal is triggered
     *
     * Pops last callback set by exec and calls it.
     */
    void on_exec();

    // for felix-register
    /**
     * @brief Get the tag to send a command to FELIX from an FID
     *
     * @param fid The tag
     * @return The tag to send the command to
     */
    [[nodiscard]] static std::uint64_t get_ctrl_fid(std::uint64_t fid);

    /**
     * @brief Get the tag to subscribe to a reply to a command from an FID
     *
     * @param ctrl_fid The tag
     * @return The tag to subscribe to
     */
    [[nodiscard]] static std::uint64_t get_subscribe_fid(std::uint64_t ctrl_fid);

    /**
    * @brief Synchronously subscribe to a set of felix-register tag
    *
    * Used by send_cmd to receive replies from felix-register.
    *
    * @throws felix::BusException If the information is not available in the bus
    * @throws felix::SubscriptionFailedError If the subscription fails
    * @param fid The tag to subscribe to
    * @param timeout The timeout for establishing the connection
    */
    void subscribe_register(const std::vector<netio_tag_t>& fids, std::chrono::milliseconds timeout);

    constexpr static auto RECONNECT_TIMER_INTERVAL = 1000ms;
    constexpr static auto TIMEOUT_CONNECT_SEND = 5000ms;

    std::string m_affinity;
    std::atomic<bool> m_terminating_felix_client;
    std::atomic<bool> m_ready{};
    std::string m_local_ip_address;

    std::shared_ptr<netio3::BaseEventLoop> m_evloop;

    OnInitCallback m_cb_on_init;
    OnDataCallback m_cb_on_data;
    OnBufferCallback m_cb_on_buffer;
    OnConnectCallback m_cb_on_connect;
    OnDisconnectCallback m_cb_on_disconnect;
    OnConnectionRefusedCallback m_cb_on_refused;
    OnSubscriptionsLostCallback m_cb_on_subscriptions_lost;
    OnResubscriptionCallback m_cb_on_resubscription;
    OnUserTimerCallback m_cb_on_user_timer;

    internal::ReconnectTimer m_reconnect_timer;
    internal::SubscriptionManager m_sub_manager;
    internal::SubscriptionManager m_sub_manager_register;
    internal::SenderManager m_sender_manager;

    netio3::EventTimerHandle m_user_timer;

    netio3::EventSignalHandle m_exec_handle;
    tbb::concurrent_queue<UserFunction> m_exec_functions;

    // felixbus
    felixbus::FelixBusReader m_bus;

    // felix-register
    simdjson::dom::parser m_parser;
    std::unordered_set<std::string> m_uuids;
    std::unordered_map<std::string, FelixClientThreadExtension::Reply> m_reply_by_uuid;

    trickle::TrickleFormatter m_trickle_formatter; ///< Formatter for trickle configurations
};

#endif // FELIXCLIENT_FELIXCLIENT_HPP