#pragma once

#include <cstdint>
#include <format>
#include <stdexcept>
#include <system_error>
#include <vector>

namespace felix {
class FelixClientException: public std::runtime_error {
public:
    explicit FelixClientException(std::string const& message) : std::runtime_error(message) {
    }
};

class MessageTooBigException: public FelixClientException {
public:
    MessageTooBigException() : FelixClientException("Message too big for buffer") {
    }
};

class SendBeforeOpenException: public FelixClientException {
public:
    SendBeforeOpenException() : FelixClientException("No send connection established. Call init_send_data first") {
    }
};

class ResourceNotAvailableException: public FelixClientException {
public:
    ResourceNotAvailableException() : FelixClientException("Resource currently not available, try again") {
    }
};

class PartiallyCompletedException: public FelixClientException {
public:
    PartiallyCompletedException() : FelixClientException("Operation completed partially, try again") {
    }
};

class SendConnectionException: public FelixClientException {
public:
    SendConnectionException() : FelixClientException("Could not establish send connection, try again") {
    }
};

class SendConnectionRefusedException: public FelixClientException {
public:
    SendConnectionRefusedException() : FelixClientException("Send connection was refused") {
    }
};

class SendConnectionTimeoutException: public FelixClientException {
public:
    SendConnectionTimeoutException() : FelixClientException("Send connection timed out") {
    }
};

class SubscribeException: public FelixClientException {
public:
    SubscribeException() : FelixClientException("Could not subscribe, try again") {
    }
};

class AlreadySubscribedException: public FelixClientException {
public:
    AlreadySubscribedException() : FelixClientException("You are already subscribed to the provided FID(s)") {
    }
};

class FailedSubscribeException: public FelixClientException {
public:
    FailedSubscribeException(
        std::vector<std::uint64_t> fids_success,
        std::vector<std::uint64_t> fids_bad,
        std::vector<std::uint64_t> fids_failed,
        std::vector<std::uint64_t> fids_timeout)
        : FelixClientException("Could not subscribe"),
          m_fids_success{std::move(fids_success)},
          m_fids_bad{std::move(fids_bad)},
          m_fids_failed{std::move(fids_failed)},
          m_fids_timeout{std::move(fids_timeout)} {}

    const std::vector<std::uint64_t>& get_fids_success() const { return m_fids_success; }
    const std::vector<std::uint64_t>& get_fids_bad() const { return m_fids_bad; }
    const std::vector<std::uint64_t>& get_fids_failed() const { return m_fids_failed; }
    const std::vector<std::uint64_t>& get_fids_timeout() const { return m_fids_timeout; }
private:
    std::vector<std::uint64_t> m_fids_success;
    std::vector<std::uint64_t> m_fids_bad;
    std::vector<std::uint64_t> m_fids_failed;
    std::vector<std::uint64_t> m_fids_timeout;
};

class FailedAsyncSubscribeException: public FelixClientException {
public:
    explicit FailedAsyncSubscribeException(std::vector<std::uint64_t> fids_bad)
        : FelixClientException("Could not subscribe"),
          m_fids_bad{std::move(fids_bad)} {}

    const std::vector<std::uint64_t>& get_fids_bad() const { return m_fids_bad; }
private:
    std::vector<std::uint64_t> m_fids_bad;
};

class NotSubscribedException: public FelixClientException {
public:
    NotSubscribedException() : FelixClientException("You are not subscribed to the provided FID") {
    }
};

class FailedUnsubscribeException: public FelixClientException {
public:
    FailedUnsubscribeException() : FelixClientException("Failed to unsubscribe") {
    }
};

class UnsubscribeTimeoutException: public FelixClientException {
public:
    UnsubscribeTimeoutException() : FelixClientException("Unsubscription timed out") {
    }
};

class BusException: public FelixClientException {
public:
    BusException(const std::uint64_t fid, const std::exception& ex) :
        FelixClientException(std::format("Bus error for FID {:#x}: {}", fid, ex.what())),
        m_fid{fid},
        m_underlying_message{ex.what()}
    {}

    [[nodiscard]] std::uint64_t get_fid() const { return m_fid; }
    [[nodiscard]] std::string get_underlying_message() const { return m_underlying_message; }
private:
    std::uint64_t m_fid{};
    std::string m_underlying_message{};
};

class SendWhileConnectionDownException: public FelixClientException {
public:
    SendWhileConnectionDownException() : FelixClientException("The send connection is currently down and being reconnected, try again") {
    }
};

class TrickleLogicException: public FelixClientException {
public:
   explicit TrickleLogicException(const std::string& er) : FelixClientException(std::format("Wrong order calling trickle funtions: {}", er)) {}
};

} // namespace felix

using FelixClientException [[deprecated("Use felix::FelixClientException instead")]] = felix::FelixClientException;
using FelixClientMessageTooBigException [[deprecated("Use felix::MessageTooBigException instead")]] = felix::MessageTooBigException;
using FelixClientSendBeforeOpenException [[deprecated("Use felix::SendBeforeOpenException instead")]] = felix::SendBeforeOpenException;
using FelixClientResourceNotAvailableException [[deprecated("Use felix::ResourceNotAvailableException instead")]] = felix::ResourceNotAvailableException;
using FelixClientPartiallyCompletedException [[deprecated("Use felix::PartiallyCompletedException instead")]] = felix::PartiallyCompletedException;
using FelixClientSendConnectionException [[deprecated("Use felix::SendConnectionException instead")]] = felix::SendConnectionException;
using FelixClientSendConnectionRefusedException [[deprecated("Use felix::SendConnectionRefusedException instead")]] = felix::SendConnectionRefusedException;
using FelixClientSendConnectionTimeoutException [[deprecated("Use felix::SendConnectionTimeoutException instead")]] = felix::SendConnectionTimeoutException;
using FelixClientSubscribeException [[deprecated("Use felix::SubscribeException instead")]] = felix::SubscribeException;
using FelixClientAlreadySubscribedException [[deprecated("Use felix::AlreadySubscribedException instead")]] = felix::AlreadySubscribedException;
using FelixClientFailedSubscribeException [[deprecated("Use felix::FailedSubscribeException instead")]] = felix::FailedSubscribeException;
using FelixClientFailedAsyncSubscribeException [[deprecated("Use felix::FailedAsyncSubscribeException instead")]] = felix::FailedAsyncSubscribeException;
using FelixClientNotSubscribedException [[deprecated("Use felix::NotSubscribedException instead")]] = felix::NotSubscribedException;
using FelixClientFailedUnsubscribeException [[deprecated("Use felix::FailedUnsubscribeException instead")]] = felix::FailedUnsubscribeException;
using FelixClientUnsubscribeTimeoutException [[deprecated("Use felix::UnsubscribeTimeoutException instead")]] = felix::UnsubscribeTimeoutException;
using FelixClientBusException [[deprecated("Use felix::BusException instead")]] = felix::BusException;
using FelixClientSendWhileConnectionDownException [[deprecated("Use felix::SendWhileConnectionDownException instead")]] = felix::SendWhileConnectionDownException;
using FelixClientTrickleLogicException [[deprecated("Use felix::TrickleLogicException instead")]] = felix::TrickleLogicException;
