#ifndef FELIXBUSFS_EXCEPTIONS_HPP
#define FELIXBUSFS_EXCEPTIONS_HPP

#include <filesystem>
#include <format>
#include <stdexcept>

namespace felixbus {
  /**
   * @brief Exception thrown when the bus is locked
   *
   * This exception is thrown when the bus is locked by another process and cannot be accessed.
   */
  class LockedException : public std::runtime_error
  {
  public:
    LockedException() : std::runtime_error("Bus is locked") {}
  };

  /**
   * @brief Exception thrown when metdata cannot be retrieved from the system
   *
   * For example, reading the user name or host name failed while writing the bus.
   */
  class FailedGetMetadataException : public std::runtime_error
  {
  public:
    explicit FailedGetMetadataException(std::string_view type) :
      std::runtime_error(std::format("Failed to get {}", type))
    {}
  };

  /**
   * @brief Exception thrown when the bus directory cannot be created
   */
  class FailedCreatingBusDirException : public std::runtime_error
  {
  public:
    FailedCreatingBusDirException(const std::filesystem::path& location, std::string_view message) :
      std::runtime_error(std::format("Failed to create bus director at {}. Underlying error: {}",
                                     location.string(),
                                     message))
    {}
  };

  /**
   * @brief Exception thrown when the bus file cannot be opened
   *
   * This exeption means, the bus does exist but cannot be opened. The underlying OS error is
   * reported,
   */
  class FailedOpeningBusException : public std::runtime_error
  {
  public:
    explicit FailedOpeningBusException(std::string_view message) :
      std::runtime_error(
        std::format("Failed opening bus (OS error). Underlying error: {}", message))
    {}
  };

  /**
   * @brief Exception thrown when the bus file cannot be written
   */
  class FailedWritingBusException : public std::runtime_error
  {
  public:
    explicit FailedWritingBusException(std::string_view message) :
      std::runtime_error(
        std::format("Failed writing bus (OS error). Underlying error: {}", message))
    {}
  };

  /**
   * @brief Exception thrown when the bus file cannot be parsed
   *
   * this exception means that the file content was not a correct JSON.
   */
  class FailedParsingBusException : public std::runtime_error
  {
  public:
    explicit FailedParsingBusException(std::string_view message) :
      std::runtime_error(
        std::format("Failed to parse bus file (JSON error). Underlying error: {}", message))
    {}
  };

  /**
   * @brief Exception thrown when the bus does not exist
   */
  class BusNotFoundException : public std::runtime_error
  {
  public:
    explicit BusNotFoundException(const std::filesystem::path& location) :
      std::runtime_error(std::format("Bus file ({}) not found", location.string()))
    {}
  };

  /**
   * @brief Exception thrown when the entry is not found in the bus
   *
   * This exception is thrown when the entry with the given FID is not found in the bus but the bus
   * does exist.
   */
  class EntryNotFoundException : public std::runtime_error
  {
  public:
    EntryNotFoundException(const std::uint64_t fid, const std::filesystem::path& location) :
      std::runtime_error(std::format("FID ({:#x}) not found in bus at {}", fid, location.string()))
    {}
  };
}  // namespace felixbus

#endif  // FELIXBUSFS_EXCEPTIONS_HPP