#ifndef FELIXBUSFS_FELIXBUSINFO_HPP
#define FELIXBUSFS_FELIXBUSINFO_HPP

#include <cstdint>
#include <ostream>
#include <iostream>
#include <string>

namespace felixbus {
  /**
   * @brief Information about an entry in the FELIX bus
   */
  struct FelixBusInfo {
    std::string host{};  //!< Hist name of the server
    pid_t pid = 0;       //!< Process ID of the server
    std::string user{};  //!< User name of user running the server

    std::uint64_t fid = 0;             //!< FID of data stream
    std::string hfid = "0x0";          //!< Human readable FID of data stream
    std::string ip{};                  //!< IP address of the server serving this FID
    std::uint16_t port = 0;            //!< Port number of the server serving this FID
    bool unbuffered = false;           //!< Whether communication is unbuffered
    bool pubsub = false;               //!< Whether communication is publish/subscribe
    bool tcp = false;                  //!< Whether communication is TCP (otherwise RDMA)
    bool stream = false;               //!< Whether the data contain stream IDs
    std::uint64_t netio_pages = 0;     //!< Number of network buffers
    std::uint64_t netio_pagesize = 0;  //!< Size of network buffers
    bool blocks = false;               //!< Whether entire blocks are sent

    constexpr auto operator<=>(const FelixBusInfo&) const = default;

    friend std::ostream& operator<<(std::ostream& os, const FelixBusInfo& info)
    {
      os << "fid             " << info.fid << '\n';
      os << "hfid            " << info.hfid << '\n';
      os << "host:           " << info.host << '\n';
      os << "pid:            " << info.pid << '\n';
      os << "user:           " << info.user << '\n';
      os << "ip:             " << info.ip << '\n';
      os << "port:           " << info.port << '\n';
      os << "unbuffered:     " << info.unbuffered << '\n';
      os << "pubsub:         " << info.pubsub << '\n';
      os << "tcp:            " << info.tcp << '\n';
      os << "stream:         " << info.stream << '\n';
      os << "netio_pages:    " << info.netio_pages << '\n';
      os << "netio_pagesize: " << info.netio_pagesize << '\n';
      return os;
    }
  };

  /**
   * @brief Keys for the FelixBusInfo struct
   */
  enum class Keys {
    pid,
    host,
    user,
    fid,
    hfid,
    ip,
    port,
    unbuffered,
    tcp,
    stream,
    pubsub,
    netio_pages,
    netio_pagesize,
    blocks,
  };

  /**
   * @brief Convert a key to a string
   *
   * @param key The key to convert
   * @return The string representation of the key
   * @throw std::invalid_argument if the key is invalid
   */
  constexpr std::string_view key_to_string(Keys key)
  {
    switch (key) {
    case Keys::pid:
      return "pid";
    case Keys::host:
      return "host";
    case Keys::user:
      return "user";
    case Keys::fid:
      return "fid";
    case Keys::hfid:
      return "hfid";
    case Keys::ip:
      return "ip";
    case Keys::port:
      return "port";
    case Keys::unbuffered:
      return "unbuffered";
    case Keys::tcp:
      return "tcp";
    case Keys::stream:
      return "stream";
    case Keys::pubsub:
      return "pubsub";
    case Keys::netio_pages:
      return "netio_pages";
    case Keys::netio_pagesize:
      return "netio_pagesize";
    case Keys::blocks:
      return "blocks";
    }
    throw std::invalid_argument("Invalid key");
  }
}  // namespace felixbus

#endif  // FELIXBUSFS_FELIXBUSINFO_HPP