#ifndef FELIXBUSFS_UTILITY_HPP
#define FELIXBUSFS_UTILITY_HPP

#include <concepts>
#include <cstring>
#include <filesystem>
#include <format>
#include <string>
#include <string_view>

namespace felixbus {
  /**
   * @brief Converts an integral value to a hexadecimal string representation
   *
   * @tparam T An integral type
   * @param val The value to convert
   * @param width The minimum width of the output string (default: sizeof(T)*2)
   * @return std::string Hexadecimal string representation of the value
   */
  template<std::integral T>
  static std::string int_to_hex(T val, size_t width = sizeof(T) << 1U)
  {
    return std::format("{0:0{1}x}", val, width);
  }

  /**
   * @brief Constructs a filesystem path for the bus
   *
   * @param bus_path_prefix The base path prefix
   * @param groupname The group name
   * @param did Device ID
   * @param cid Component ID
   * @return std::filesystem::path Constructed path in format:
   * <prefix>/<groupname>/<did_hex>/<cid_hex>
   */
  [[nodiscard]] inline std::filesystem::path get_path(const std::string_view bus_path_prefix,
                                                      const std::string_view groupname,
                                                      const std::uint8_t did,
                                                      const std::uint32_t cid)
  {
    // for future compatibility, DID and CID are encoded as shortest hexadecimal, VID is only used
    // in decoding the FID
    return std::filesystem::path(bus_path_prefix) / groupname / int_to_hex(did, 1) /
           int_to_hex(cid, 1);
  }

  /**
   * @brief Constructs a full filename path for a bus file
   *
   * @param bus_path_prefix The base path prefix
   * @param groupname The group name
   * @param did Device ID
   * @param cid Component ID
   * @param bus_filename The base filename
   * @return std::filesystem::path Constructed path with .ndjson extension
   */
  [[nodiscard]] inline std::filesystem::path get_filename(const std::string_view bus_path_prefix,
                                                          const std::string_view groupname,
                                                          const std::uint8_t did,
                                                          const std::uint32_t cid,
                                                          const std::string_view bus_filename)
  {
    return (get_path(bus_path_prefix, groupname, did, cid) / bus_filename)
      .replace_extension(".ndjson");
  }

  /**
   * @brief Get a string representation of an error number
   *
   * @param errnum The error number
   * @return std::string The error message
   */
  [[nodiscard]] inline std::string get_str_error(int errnum)
  {
    constexpr static auto BUFFER_SIZE = std::size_t{256};
    std::array<char, BUFFER_SIZE> buffer{};
    if constexpr (requires(int err, char* buf, size_t buflen) {
                    { strerror_r(err, buf, buflen) } -> std::convertible_to<char*>;
                  }) {
      // GNU-specific version
      char* message = strerror_r(errnum, buffer.data(), buffer.size());
      if (message != nullptr) {
        return {message};
      }
    } else if constexpr (requires(int err, char* buf, size_t buflen) {
                           { strerror_r(err, buf, buflen) } -> std::convertible_to<int>;
                         }) {
      // POSIX-compliant version
      if (strerror_r(errnum, buffer.data(), buffer.size()) == nullptr) {
        return {buffer.data()};
      }
    }
    return "Unknown error";
  }
}  // namespace felixbus

#endif  // FELIXBUSFS_FELIXBUS_HPP