#ifndef NETIO3BACKEND_EVENTLOOP_EPOLLEVENTLOOP_HPP
#define NETIO3BACKEND_EVENTLOOP_EPOLLEVENTLOOP_HPP

#include <atomic>
#include <functional>
#include <map>
#include <mutex>
#include <unordered_map>

#include <fcntl.h>
#include <sys/epoll.h>
#include <sys/eventfd.h>
#include <sys/timerfd.h>
#include <unistd.h>

#include <tbb/concurrent_queue.h>

#include "netio3-backend/EventLoop/EventSignalHandle.hpp"
#include "netio3-backend/EventLoop/BaseEventLoop.hpp"

namespace netio3 {
  /**
   * @brief The AsioEventLoop class represents an event loop using epoll_wait
   *
   * This class uses a native linux epoll_wait instance as the event loop driver. File descriptors
   * are registered with epoll_ctl.
   */
  class EpollEventLoop : public BaseEventLoop
  {
  public:
    explicit EpollEventLoop(std::function<void()> cb_init = nullptr);
    ~EpollEventLoop() override;
    EpollEventLoop(const EpollEventLoop& other) = delete;
    EpollEventLoop& operator=(const EpollEventLoop& other) = delete;
    EpollEventLoop(EpollEventLoop&& other) = delete;
    EpollEventLoop& operator=(EpollEventLoop&& other) = delete;

    /**
     * @copydoc BaseEventLoop::run
     *
     * Closes all remaining fd when terminating. While stopping waits up to 5 iterations to finish
     * callbacks before force stopping.
     */
    void run() override;
    
    /**
     * @copydoc BaseEventLoop::run_for
     */
    void run_for(uint64_t t_sec) override;

    /**
     * @copydoc BaseEventLoop::register_fd
     *
     * The file descriptor is not registered immediately but queued and a signal is fired to
     * actually register it. This procedure is used to have a lock free but thread safe registration
     * of file descriptors.
     */
    void register_fd(const EventContext& ctx) override;

    /**
     * @copydoc BaseEventLoop::remove_fd
     *
     * The file descriptor is not removed immediately but queued and a signal is fired to actually
     * remove it. This procedure is used to have a lock free but thread safe removal of file
     * descriptors.
     */
    void remove_fd(int fd, bool close_fd = false, bool wait = false) override;

    /**
     * @copydoc BaseEventLoop::stop
     *
     * The event loop is stopped by setting a flag. The actual stopping is done in the event loop.
     * This allows outstanding events to be processed before stopping.
     */
    void stop() override;
    
    /**
     * @copydoc BaseEventLoop::is_running
     */
    [[nodiscard]] bool is_running() const override;

  private:
    constexpr static auto MAX_EPOLL_EVENTS = 128;  ///< Maximum number of events to process
    constexpr static auto EPOLL_TIMEOUT = 1000;  ///< Timeout for epoll_wait in milliseconds
    constexpr static auto MIN_POLL_TIMEOUT = 10;  ///< Timeout for epoll_wait when stopping

    /**
      * @brief Struct holding fd and flag wheter to call close on it
     */
    struct CloseFdQueueItem {
      int fd{};
      bool close_fd{};
      bool wait{};
    };

    /**
     * @brief Process the given event context
     *
     * Call the callback registered with a file descriptor if a callback was registered.
     *
     * @param evc The event context to process
     */
    static void process_event(const EventContext& evc);

    /**
     * @brief Registers all queued file descriptors
     *
     * Actually register file descriptors that are queued for registration by @ref register_fd.
     */
    void register_queued_fd();

    /**
     * @brief Unregisters all queued file descriptors
     *
     * Actually remove file descriptors that are queued for removal by @ref register_fd. Call close
     * if this was requested.
     */
    void unregister_queued_fd();

    /**
     * @brief Actually register a file descriptor
     *
     * Register a file descriptor with epoll_ctl. The file descriptor is set to non-blocking mode.
     *
     * If track_fd is true, the file descriptor is added to the list of open file descriptors. All
     * open file descriptors are closed when the event loop is stopped. Some file descriptors are
     * bound to the life time of the event loop and should use track_fd = false (e.g. the signal to
     * register fds).
     *
     * @param ctx The event context to register
     * @param track_fd Whether to track the file descriptor
     */
    void do_register_fd(const EventContext& ctx, bool track_fd = true);

    /**
     * @brief Actually unregister a file descriptor
     *
     * Unregister a file descriptor with epoll_ctl.
     *
     * @param fd The file descriptor to unregister
     * @param close_fd Whether to close the file descriptor
     */
    void do_unregister_fd(int fd, bool close_fd);

    /**
     * @brief Create an internal signal
     *
     * The event loop has two signals to register and unregister file descriptors. These cannot be
     * created using create_signal. This function has to be used instead.
     *
     * @param cb The callback to execute when the signal is fired
     * @return The file descriptor of the signal
     */
    [[nodiscard]] int create_internal_signal(const std::function<void(int)>& cb);

    /**
     * @brief Fire an internal signal
     *
     * Since @ref create_internal_signal does not return a handle, this function is used to fire
     * internal signals.
     *
     * @param fd The file descriptor of the signal
     */
    static void fire_internal_signal(int fd);

    /**
     * @brief Stop the event loop
     *
     * This function is used to stop the event loop. It contains the actual implementation in a
     * non-virtual function.
     */
    void do_stop();

    std::map<int, EventContext> m_ev_contexts_by_fd;
    int m_epollfd{};
    std::function<void()> m_cb_init{nullptr};
    std::vector<int> m_open_fds;
    std::vector<int> m_fds_to_remove;
    std::vector<EventContext> m_ev_contexts;
    tbb::concurrent_queue<EventContext> m_register_queue;
    tbb::concurrent_queue<CloseFdQueueItem> m_unregister_queue;
    std::unordered_map<int, bool> m_fds_to_close;
    int m_register_fd_signal{};
    int m_unregister_fd_signal{};
    int m_stop_fd_signal{};
    std::atomic_bool m_ev_should_stop = false;
    std::atomic_bool m_is_running = false;
    mutable std::mutex m_mutex;
  };

}  // namespace netio3

#endif  // NETIO3BACKEND_EVENTLOOP_EPOLLEVENTLOOP_HPP