#ifndef NETIO3BACKEND_EVENTLOOP_EVENTSIGNALHANDLE_HPP
#define NETIO3BACKEND_EVENTLOOP_EVENTSIGNALHANDLE_HPP

#include <memory>

#include <ers/ers.h>

ERS_DECLARE_ISSUE(netio3,
                  FailedFireSignal,
                  std::format("Firing signal writing on fd {}, only {} / 8 bytes written. Errno {}",
                              fd,
                              ret,
                              message),
                  ((int)fd)((ssize_t)ret)((const std::string&)message));

namespace netio3 {
  class BaseEventLoop;

  /**
   * @brief The EventSignalHandle class represents a signal in the event loop
   *
   * An EventSignalHandle is created by calling @ref create_signal. It offers an interface to fire
   * the signal and acts as an RAII handle to the signal. When the handle is destroyed, the signal
   * is removed from the event loop.
   *
   * The event loop does guarantee that outstanding events are handled up to a timeout even if the
   * signal is removed.
   */
  class EventSignalHandle
  {
  public:
    /**
     * @brief Construct a new EventSignalHandle object
     *
     * Not supposed to be called directly. Use @ref create_signal instead.
     *
     * @param fd The file descriptor of the signal
     * @param evloop The event loop to register the signal with
     */
    explicit EventSignalHandle(int fd, const std::shared_ptr<BaseEventLoop>& evloop);

    /**
     * @brief Destroy the EventSignalHandle object
     *
     * If this is the last reference to the signal, the signal is removed from the event loop.
     */
    ~EventSignalHandle();
    EventSignalHandle(const EventSignalHandle&) = default;
    EventSignalHandle(EventSignalHandle&&) = default;
    EventSignalHandle& operator=(const EventSignalHandle&) = default;
    EventSignalHandle& operator=(EventSignalHandle&&) = default;

    /**
     * @brief Fires the signal
     *
     * Writes a byte to the file descriptor of the signal. If the write fails, an error is issued.
     */
    void fire() const;

    /**
     * @brief Get the file descriptor of the signal
     *
     * @return The file descriptor of the signal
     */
    [[nodiscard]] int get_fd() const { return *m_fd; }

  private:
    std::shared_ptr<int> m_fd{};
    std::weak_ptr<BaseEventLoop> m_evloop;
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_EVENTLOOP_EVENTSIGNALHANDLE_HPP