#ifndef NETIO3BACKEND_EVENTLOOP_EVENTTIMERHANDLE_HPP
#define NETIO3BACKEND_EVENTLOOP_EVENTTIMERHANDLE_HPP

#include <chrono>

#include <sys/timerfd.h>

#include <ers/ers.h>

#include "netio3-backend/EventLoop/Utility.hpp"

ERS_DECLARE_ISSUE(netio3, FailedSetTimer, std::format("Failed to set timer for FD {}: {}", fd, message), ((int) fd)((const std::string&) message))

namespace netio3 {
  class BaseEventLoop;

  /**
   * @concept ChronoDuration
   * @brief Concept to check if a type is a chrono duration
   * @tparam T The type to check
   */
  template<typename T>
  concept ChronoDuration =
    std::is_same_v<std::decay_t<T>, std::chrono::duration<typename T::rep, typename T::period>>;

  /**
   * @brief The EventTimerHandle class represents a timer in the event loop
   *
   * An EventTimerHandle is created by calling @ref create_timer. It offers an interface to start
   * and stop the timer. When the handle is destroyed, the timer is removed from the event loop.
   *
   * The frequency of the timer is passed to the start function. It will be periodically called with
   * the given frequency. The first call happens after the duration has elapsed once.
   */
  class EventTimerHandle
  {
  public:
    /**
     * @brief Construct a new EventTimerHandle object
     *
     * Not supposed to be called directly. Use @ref create_timer instead.
     *
     * @param fd The file descriptor of the timer
     * @param evloop The event loop to register the timer with
     */
    explicit EventTimerHandle(int fd, const std::shared_ptr<BaseEventLoop>& evloop);

    /**
     * @brief Destroy the EventTimerHandle object
     *
     * If this is the last reference to the timer, the timer is removed from the event loop.
     */
    ~EventTimerHandle();
    EventTimerHandle(const EventTimerHandle&) = default;
    EventTimerHandle(EventTimerHandle&&) = default;
    EventTimerHandle& operator=(const EventTimerHandle&) = default;
    EventTimerHandle& operator=(EventTimerHandle&&) = default;

    /**
     * @brief Starts the timer
     *
     * The timer is started with the given duration. The first call happens after the duration has
     * elapsed once.
     *
     * @param duration The duration to start the timer with
     */
    void start(const ChronoDuration auto& duration)
    {
      const auto ts = convert_duration(duration);
      const auto its = itimerspec{.it_interval = ts, .it_value = ts};
      if (timerfd_settime(*m_fd, 0, &its, nullptr) == -1) {
        ers::error(FailedSetTimer(ERS_HERE, *m_fd, utility::error_message(errno)));
      } else {
        m_is_running = true;
      }
    }

    /**
     * @brief Stops the timer
     */
    void stop();

    /**
     * @brief Get the file descriptor of the timer
     *
     * @return The file descriptor of the timer
     */
    [[nodiscard]] int get_fd() const { return *m_fd; }

    /**
     * @brief Check if the timer is running
     *
     * @return True if the timer is running, false otherwise
     */
    [[nodiscard]] bool is_running() const { return m_is_running; }

  private:
    /**
     * @brief Convert a duration to a timespec
     *
     * @param duration The duration to convert
     * @return The timespec representation of the duration
     */
    [[nodiscard]] static timespec convert_duration(const ChronoDuration auto& duration)
    {
      const auto s = std::chrono::duration_cast<std::chrono::seconds>(duration);
      const auto ns = std::chrono::duration_cast<std::chrono::nanoseconds>(duration - s);
      return timespec{.tv_sec = s.count(), .tv_nsec = ns.count()};
    }

    std::shared_ptr<int> m_fd{};
    bool m_is_running{false};
    std::weak_ptr<BaseEventLoop> m_evloop;
  };
}  // namespace netio3

#endif  // NETIO3BACKEND_EVENTLOOP_EVENTTIMERHANDLE_HPP