#ifndef NETIO3BACKEND_NETIO3BACKEND_HPP
#define NETIO3BACKEND_NETIO3BACKEND_HPP

#include <cstring>
#include <iostream>
#include <map>
#include <memory>
#include <stdexcept>
#include <vector>

#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/eventfd.h>

#include "netio3-backend/EventLoop/BaseEventLoop.hpp"
#include "netio3-backend/NetworkBuffer.hpp"

namespace netio3 {
  enum class NetworkMode { TCP, RDMA, RDM };
  enum class NetworkType { LIBFABRIC, ASYNCMSG };
  enum class NetioStatus { OK, NO_RESOURCES, FAILED };
  enum class ThreadSafetyModel {
    SAFE,
    UNSAFE,
  };

  /**
   * @brief The number of header slots in the buffer for zero-copy sending
   *
   * This number corresponds to the maximum number of concurrent zero-copy sends. The allocated
   * buffer is equal to this times @ref SIZE_HEADER.
   */
  constexpr static auto ZERO_COPY_NUM_HEADER_SLOTS{512};
  /**
   * @brief The maximum size of user data in the header for zero-copy sending
   */
  constexpr static auto ZERO_COPY_MAX_SIZE_USER_DATA{8};
  /**
   * @brief The size of the header in the buffer for zero-copy sending
   *
   * The header size is calculated as the sum of the size of a tag (8 byte) and the maximum size of
   * user data.
   */
  constexpr static auto ZERO_COPY_SIZE_HEADER{sizeof(std::uint64_t) + ZERO_COPY_MAX_SIZE_USER_DATA};

  inline static NetworkMode network_mode_from_string(const std::string& mode)
  {
    const auto network_mode_value = std::map<std::string, NetworkMode>{
      {"TCP", NetworkMode::TCP},
      {"RDMA", NetworkMode::RDMA},
      {"RDM", NetworkMode::RDM},
    };
    if (network_mode_value.contains(mode)) {
      return network_mode_value.at(mode);
    }
    throw std::invalid_argument(std::format("network_mode: {}", mode));
  }

  inline static NetworkType network_type_from_string(const std::string& type)
  {
    const auto network_type_value = std::map<std::string, NetworkType>{
      {"LIBFABRIC", NetworkType::LIBFABRIC},
      {"ASYNCMSG", NetworkType::ASYNCMSG},
    };
    if (network_type_value.contains(type)) {
      return network_type_value.at(type);
    }
    throw std::invalid_argument(std::format("network_type: {}", type));
  }

  /**
   * @brief Class containing the IP address and port of a network endpoint
   *
   * The IP is stored as a string and as a numeric value for faster comparisons.
   */
  class EndPointAddress
  {
  public:
    EndPointAddress() = default;

    /**
     * @brief Construct an EndPointAddress object from an IP address and port
     *
     * @param ip IP address
     * @param port Port number
     */
    EndPointAddress(const std::string& ip, unsigned short port);

    /**
     * @brief Construct an EndPointAddress object from a string containing the IP address and port
     *
     * Format of the string: "IP:port" or "IP" (assumes port 0)
     *
     * @param str String containing the IP address and port
     */
    explicit EndPointAddress(const std::string& str);

    /**
     * @brief Construct an EndPointAddress object from a sockaddr structure
     *
     * This constructor is used to create an EndPointAddress from a sockaddr structure,
     * which is typically used in network programming to represent an endpoint address.
     *
     * @param addr Pointer to a sockaddr structure containing the endpoint address
     */
    explicit EndPointAddress(const sockaddr* addr, std::size_t addrlen);

    /**
     * @brief Get the IP address
     *
     * @return IP address
     */
    [[nodiscard]] const std::string& address() const { return m_ip; }

    /**
     * @brief Get the port
     *
     * @return Port number
     */
    [[nodiscard]] unsigned short port() const { return m_port; }

    /**
     * @brief Get the IP address as a numeric value
     *
     * For IPv4 addresses, returns the actual 32-bit network address extended to 64-bit.
     * For IPv6 addresses, returns a cryptographically robust 64-bit hash representation
     * with the high bit set to distinguish from IPv4.
     *
     * The 64-bit representation provides excellent collision resistance:
     * - Collision probability ~50% only after 5.1 billion different addresses
     * - Suitable for production use with millions of network endpoints
     *
     * @return IP address as a 64-bit numeric value suitable for indexing and comparison
     */
    [[nodiscard]] std::uint64_t address_numeric() const { return m_ip_numeric; }

    /**
     * @brief Set the IP address
     *
     * @param ip IP address
     */
    void port(unsigned short port) { m_port = port; }

    /**
     * @brief Convert the EndPointAddress to a sockaddr_storage structure
     *
     * Returns a sockaddr_storage that can hold both IPv4 and IPv6 addresses,
     * making it compatible with libfabric which accepts sockaddr, sockaddr_in,
     * and sockaddr_in6. The returned structure can be safely cast to the
     * appropriate type based on the sa_family field.
     *
     * @return sockaddr_storage structure representing the endpoint address
     * @throws std::invalid_argument if the IP address format is invalid
     */
    [[nodiscard]] sockaddr_storage to_sockaddr_storage() const;

    /**
     * @brief Compare two EndPointAddress objects using numeric IP and port
     *
     * @param other EndPointAddress object to compare with
     * @return Comparison result
     */
    [[nodiscard]] auto operator<=>(const EndPointAddress& other) const
    {
      return std::tie(m_ip_numeric, m_port) <=> std::tie(other.m_ip_numeric, other.m_port);
    }

    /**
     * @brief Compare two EndPointAddress objects for equality using numeric IP and port
     *
     * @param other EndPointAddress object to compare with
     * @return Comparison result
     */
    [[nodiscard]] bool operator==(const EndPointAddress& other) const
    {
      return std::tie(m_ip_numeric, m_port) == std::tie(other.m_ip_numeric, other.m_port);
    }

    /**
     * @brief Compare two EndPointAddress objects for inequality using numeric IP and port
     *
     * @param other EndPointAddress object to compare with
     * @return Comparison result
     */
    [[nodiscard]] bool operator!=(const EndPointAddress& other) const { return !(*this == other); }

    /**
     * @brief Overload the << operator to print the IP address and port
     *
     * @param stream Output stream
     * @param val EndPointAddress object
     * @return Output stream
     */
    friend std::ostream& operator<<(std::ostream&, const EndPointAddress&);

    /**
     * @brief Enable hashing with absl::Hash
     *
     * @tparam H Hash state type
     * @param h Current hash state
     * @param endpoint EndPointAddress to hash
     * @return Updated hash state
     */
    template<typename H>
    friend H AbslHashValue(H h, const EndPointAddress& endpoint)
    {
      return H::combine(std::move(h), endpoint.address_numeric(), endpoint.port());
    }

    /**
     * @brief Check if the address is IPv4
     *
     * @return true if the address is IPv4, false if IPv6
     */
    [[nodiscard]] bool is_ipv4() const;

    /**
     * @brief Check if the address is IPv6
     *
     * @return true if the address is IPv6, false if IPv4
     */
    [[nodiscard]] bool is_ipv6() const;

  private:
    /**
     * @brief Convert an IP address from string to numeric representation
     *
     * @throws std::invalid_argument if the IP address is invalid
     * @param ip IP address as a string
     * @return IP address as a numeric value
     */
    [[nodiscard]] static std::uint64_t convert_ip(const std::string& ip);

    std::string m_ip{};
    std::uint64_t m_ip_numeric{};
    unsigned short m_port{};
  };

  /**
   * @brief Hash function for EndPointAddress objects
   *
   * The hash function is used to create a hash value for EndPointAddress objects. The hash value is
   * used to store the objects in a hash table.
   */
  struct EndPointAddressHash {
    std::size_t operator()(const EndPointAddress& endpoint) const {
        std::size_t h1 = std::hash<std::uint64_t>{}(endpoint.address_numeric());
        std::size_t h2 = std::hash<unsigned short>{}(endpoint.port());
        return h1 ^ (h2 << 1);
    }
  };

  /**
   * @brief Connection parameters for buffered sending for the network backend
   *
   * The parameters are used to configure the network connection. Provides the buffer size and
   * number of buffers to be used for buffered sending. If the use_shared_send_buffers is true,
   * the buffers are shared between multiple connections when providing at backend construction.
   * Otherwise, each connection has its own buffers.
   */
  struct ConnectionParametersSendBuffered {
    size_t buf_size{};
    size_t num_buf{};
    bool use_shared_send_buffers{false};

    constexpr auto operator<=>(const ConnectionParametersSendBuffered&) const = default;
  };

  /**
   * @brief Connection parameters for zero-copy sending for the network backend
   *
   * The parameters are used to configure the network connection. Provides the memory region start
   * address and buffer size.
   */
  struct ConnectionParametersSendZeroCopy {
    size_t buf_size{};
    std::uint8_t* mr_start{nullptr};
    bool use_shared_send_buffers{false};

    constexpr auto operator<=>(const ConnectionParametersSendZeroCopy&) const = default;
  };

  /**
   * @brief Connection parameters for the network backend for receiving
   */
  struct ConnectionParametersRecv {
    std::size_t buf_size{};
    std::size_t num_buf{};
    bool use_shared_receive_buffers{false};

    auto operator<=>(const ConnectionParametersRecv&) const = default;
  };

  /**
   * @brief Struct holding all connection parameters for the network backend
   *
   * Contains the receive parameters, buffered send parameters and zero-copy send parameters.
   * The receive parameters are used to configure the receive buffers. The buffered send parameters
   * are used to configure the buffered sending. The zero-copy send parameters are used to configure
   * the zero-copy sending.
   *
   * For buffered sending and receiving, if the number of buffers is 0, the capability is not
   * requested. For zro-copy sending, the mr_start being nullptr indicates that the capability is
   * not requested.
   */
  struct ConnectionParameters {
    ConnectionParametersRecv recv_params{};
    ConnectionParametersSendBuffered send_buffered_params{};
    ConnectionParametersSendZeroCopy send_zero_copy_params{};
  };

  /**
   * @brief Struct holding the capabilities of an endpoint
   *
   * The capabilities are used to determine which operations are supported by the endpoint. The
   * capabilities are used to determine whether the endpoint can send buffered, send copy, send
   * zero-copy and receive data.
   */
  struct EndpointCapabilities {
    bool send_buffered{false};
    bool send_copy{false};
    bool send_zero_copy{false};
    bool receive{false};
  };

  // Callbacks
  using OnDataCb = std::function<void(std::span<const std::uint8_t>)>;
  using OnConnectionEstablishedCb = std::function<void(const EndPointAddress&, const EndPointAddress&, EndpointCapabilities)>;
  using OnConnectionClosedKeysCb = std::function<void(const EndPointAddress&, const std::vector<std::uint64_t>&)>;
  using OnConnectionClosedCb = std::function<void(const EndPointAddress&)>;
  using OnConnectionRefusedCb = std::function<void(const EndPointAddress&)>;
  using OnSendCompleted = std::function<void(const EndPointAddress&, uint64_t)>;

  /**
   * @brief Struct holding all callbacks for the network backend
   *
   * - on_data_cb: When data is received. The data is passed as a span of const uint8_t
   * - on_connection_established_cb: When a connection is established. Provides the address of the
   *   endpoint
   * - on_connection_closed_cb: When a connection is closed. Provides the address of the endpoint
   *   and (for zero-copy sends) the keys of the pending send operations (otherwise possibly empty)
   * - on_connection_refused_cb: When a connection is refused. Provides the address of the endpoint
   * - on_send_completed_cb: When a send operation is completed. Provides the address of the
   *   endpoint and the key of the send operation
   */
  struct CallbacksConfig {
    OnDataCb on_data_cb{nullptr};
    OnConnectionEstablishedCb on_connection_established_cb{nullptr};
    OnConnectionClosedKeysCb on_connection_closed_cb{nullptr};
    OnConnectionRefusedCb on_connection_refused_cb{nullptr};
    OnSendCompleted on_send_completed_cb{nullptr};
  };

  /**
   * @brief Configuration for the network backend
   *
   * Contains network mode, thread safety model, @ref ConnectionParametersRecv for possibly shared
   * receive buffers and @ref CallbacksConfig.
   */
  struct NetworkConfig {
    NetworkMode mode{NetworkMode::RDMA};
    ThreadSafetyModel thread_safety{ThreadSafetyModel::SAFE};
    ConnectionParameters conn_params{};
    CallbacksConfig callbacks;
    friend std::ostream& operator<<(std::ostream& stream, const NetworkConfig& cfg);
  };

  /**
   * @brief Abstract base class for network backends
   *
   * Defines the interface for all implementations of network backends. The backend is responsible
   * for managing the network connections and sending and receiving data.
   *
   * Use the factory method create to create a network backend object.
   *
   * Invokes the following callbacks:
   * - on_data_cb: When data is received. The data is passed as a span of const uint8_t
   * - on_connection_established_cb: When a connection is established. Provides the address of the
   *   endpoint
   * - on_connection_closed_cb: When a connection is closed. Provides the address of the endpoint
   *   and for zero-copy sends the keys of the pending send operations (otherwise empty)
   * - on_connection_refused_cb: When a connection is refused. Provides the address of the endpoint
   * - on_send_completed_cb: When a send operation is completed. Provides the address of the
   *   endpoint and the key of the send operation
   */
  class NetworkBackend
  {
  public:
    /**
     * @brief Constructs a NetworkBackend object with the specified network configuration and event
     * loop
     *
     * The network configuration decides whether RDMA or TCP is used. It also decides whether the
     * backend provides thread safety or not. If the backend is not thread safe, it is the user's
     * responsibility to ensure that only one thread interacts with the backend. The configuration
     * also contains the callbacks for connection events, send completions and data reception. The
     * event loop must be an implementation of the BaseEventLoop interface. The event loop is used
     * to register all kinds of asynchronous events.
     *
     * @param config The network configuration
     * @param evloop The event loop to use for network operations
     */
    NetworkBackend(NetworkConfig config, std::shared_ptr<BaseEventLoop> evloop);

    virtual ~NetworkBackend() = default;
    NetworkBackend(const NetworkBackend&) = delete;
    NetworkBackend(NetworkBackend&&) = delete;
    NetworkBackend& operator=(const NetworkBackend&) = delete;
    NetworkBackend& operator=(NetworkBackend&&) = delete;

    /**
     * @brief Factory method to create a network backend object
     *
     * Creates a network backend object based on the network type and configuration. The event loop
     * and config is passed as a parameter to the constructor of the network backend. The network
     * type decides which network backend to create.
     *
     * @param type The type of the network backend
     * @param config The network configuration
     * @param evloop The event loop to use for network operations
     * @return A unique pointer to the network backend object
     */
    static std::unique_ptr<NetworkBackend> create(NetworkType type,
                                                  const NetworkConfig& config,
                                                  const std::shared_ptr<BaseEventLoop>& evloop);

    /**
     * @brief Opens an active endpoint for the specified address with the given connection
     * parameters
     *
     * The active endpoint will connect to a listener on the specified address and port. Its
     * capabilities are defined by the connection parameters.
     *
     * @param address The address of the endpoint to open (remote address)
     * @param connection_params The connection parameters for the endpoint
     * @throws InvalidEndpointAddress if the endpoint address is invalid
     * @throws FailedOpenActiveEndpoint if the active endpoint could not be opened
     * @throws ActiveEndpointAlreadyExists if the active endpoint already exists
     * @throws InvalidConnectionParameters if the connection parameters conflict with global
     * settings (shared buffers)
     */
    virtual void open_active_endpoint(const EndPointAddress& address,
                                      const ConnectionParameters& connection_params) = 0;

    /**
     * @brief Opens a listen endpoint for the specified address with the given connection
     * parameters
     *
     * Should accept all incoming connections. The connection parameters are passed to the spawned
     * active endpoint. If port 0 was passed in, it should return the port number that was used to
     * open the listen endpoint.
     *
     * @param address The address of the endpoint to open (local address)
     * @param connection_params The connection parameters for spawned endpoints
     * @throws InvalidEndpointAddress if the endpoint address is invalid
     * @throws ListenEndpointAlreadyExists If already requested to listen on this address and port
     * @throws FailedOpenListenEndpoint if the listen endpoint could not be opened
     * @throws InvalidConnectionParameters if the connection parameters conflict with global
     * settings (shared buffers)
     * @return The actual address the server is listening on
     */
    virtual EndPointAddress open_listen_endpoint(
      const EndPointAddress& address,
      const ConnectionParameters& connection_params) = 0;

    /**
     * @brief Closes the active endpoint for the specified address
     *
     * Can be used to close both active endpoints directly opened by calling @ref
     * open_active_endpoint and endpoints that were spawned by a listen endpoint.
     *
     * May enqueue the endpoint to be closed by the event loop for thread synchronization purposes.
     *
     * @param address The address of the endpoint to close
     * @throws InvalidEndpointAddress if the endpoint address is invalid
     * @throws UnknownActiveEndpoint if the endpoint does not exist
     * @throws FailedCloseActiveEndpoint if closing the active endpoint failed
     */
    virtual void close_active_endpoint(const EndPointAddress& address) = 0;

    /**
     * @brief Closes the listen endpoint for the specified address
     *
     * Shall also close all active endpoints that were spawned by this listen endpoint.
     *
     * May enqueue the endpoint to be closed by the event loop for thread synchronization purposes.
     *
     * @param address The address of the endpoint to close
     * @throws InvalidEndpointAddress if the endpoint address is invalid
     * @throws UnknownListenEndpoint if the listen endpoint does not exist
     * @throws FailedCloseListenEndpoint if closing the listen endpoint failed
     */
    virtual void close_listen_endpoint(const EndPointAddress& address) = 0;

    /**
     * @brief Sends data to the specified address in zero-copy mode
     *
     * The data shall not be copied but sent directly. The key will be provided on the
     * on_send_completed callback to the user.
     *
     * Header data is prepended to the actual data and sent together with it.
     *
     * @param address The address to send the data to
     * @param data The data to send
     * @param header_data User data to be added to the header
     * @param key Key to be returned in on_send_completed callback
     * @throws InvalidEndpointAddress If the address is invalid
     * @throws UnknownActiveEndpoint if the endpoint does not exist or has no send capabilities
     * @return The status of the send operation
     */
    virtual NetioStatus send_data(const EndPointAddress& address,
                                  std::span<std::uint8_t> data,
                                  std::span<const std::uint8_t> header_data,
                                  std::uint64_t key) = 0;

    /**
     * @brief Sends data to the specified address in zero-copy mode
     *
     * The data shall not be copied but sent directly. The key will be provided on the
     * on_send_completed callback to the user.
     *
     * Header data is prepended to the actual data and sent together with it.
     *
     * @param address The address to send the data to
     * @param iov The iovec vector containing the data to send
     * @param header_data User data to be added to the header
     * @param key Key to be returned in on_send_completed callback
     * @throws InvalidEndpointAddress If the address is invalid
     * @throws UnknownActiveEndpoint if the endpoint does not exist or has no send capabilities
     * @return The status of the send operation
     */
    virtual NetioStatus send_data(const EndPointAddress& address,
                                  std::span<const iovec> iov,
                                  std::span<const std::uint8_t> header_data,
                                  std::uint64_t key) = 0;

    /**
      * @brief Sends data to the specified address copying the data
      *
      * The data are copied and sent directly. The user does not have to make any guarantees about
      * the lifetime of the data. The key will be provided on the on_send_completed callback to the
      * user. The header data is prepended to the actual data and sent together with it.
      *
      * @param address The address to send the data to
      * @param data The data to send
      * @param header_data User data to be added to the header
      * @param key Key to be returned in on_send_completed callback
      * @return The status of the send operation
      */
    virtual NetioStatus send_data_copy(const EndPointAddress& address,
                                       std::span<const std::uint8_t> data,
                                       std::span<const std::uint8_t> header_data,
                                       std::uint64_t key) = 0;

    /**
      * @brief Sends data to the specified address in zero-copy mode
      *
      * The data are copied and sent directly. The user does not have to make any guarantees about
      * the lifetime of the data. The key will be provided on the on_send_completed callback to the
      * user. The header data is prepended to the actual data and sent together with it.
      *
      * @param address The address to send the data to
      * @param iov The iovec vector containing the data to send
      * @param header_data User data to be added to the header
      * @param key Key to be returned in on_send_completed callback
      * @return The status of the send operation
      */
    virtual NetioStatus send_data_copy(const EndPointAddress& address,
                                       std::span<const iovec> iov,
                                       std::span<const std::uint8_t> header_data,
                                       std::uint64_t key) = 0;

    /**
     * @brief Retrieves a network buffer for the specified endpoint address
     *
     * Only works for endpoints that are opened for buffered sending. If no buffer is available
     * returns a nullptr.
     *
     * @important The user needs to check that the returned buffer is not null.
     *
     * @param address The address of the endpoint to retrieve the buffer for
     * @throws UnknownActiveEndpoint if the endpoint does not exist or has no send capabilities
     * @throws NoBuffersAllocated if no buffers were allocated
     * @return A pointer to the network buffer
     */
    virtual NetworkBuffer* get_buffer(const EndPointAddress& address) = 0;

    /**
     * @brief Sends a network buffer to the specified address
     *
     * This function sends the data buffer to the specified endpoint. A connection must be
     * registered for this endpoint before sending data. If the buffer is not a buffer handed out
     * through @ref get_buffer FAILED is returned. The buffer will be returned to the connection
     * after the send operation is completed.
     *
     * @param address The address to send the buffer to
     * @param buffer The network buffer to send
     * @throws UnknownActiveEndpoint if the endpoint does not exist or has no send capabilities
     * @return The status of the send operation
     */
    virtual NetioStatus send_buffer(const EndPointAddress& address, NetworkBuffer* buffer) = 0;

    /**
     * @brief Retrieves the number of available buffers for the specified endpoint address
     *
     * Returns the mininum number of available buffers since the last call to this function.
     *
     * @param address The address of the endpoint to retrieve the buffer count for
     * @return The number of available buffers
     */
    virtual std::size_t get_num_available_buffers(const EndPointAddress& address) = 0;

  protected:
    NetworkConfig m_config;
    std::shared_ptr<BaseEventLoop> m_evloop;
  };
};  // namespace netio3

#endif  // NETIO3BACKEND_NETIO3BACKEND_HPP