#ifndef NETIO3_NETIOPUBLISHER_HPP
#define NETIO3_NETIOPUBLISHER_HPP

#include <cstdint>
#include <cstdlib>
#include <cstring>
#include <functional>
#include <memory>
#include <map>
#include <set>
#include <span>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#include <vector>

#include <ers/ers.h>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/EventLoop/EventTimerHandle.hpp>
#include <netio3-backend/Issues.hpp>
#include <netio3-backend/Netio3Backend.hpp>

#include "netio3/SubscriptionEndpointContainer/Definition.hpp"
#include "netio3/SubscriptionEndpointContainer/SubscriptionEndpointMap.hpp"
#include "netio3/MonitoringStats.hpp"
#include "netio3/NetioIssues.hpp"
#include "netio3/NetioReceiver.hpp"
#include "netio3/NetioSender.hpp"
#include "netio3/SubscriptionRequest.hpp"
#include "netio3/Types.hpp"
#include "netio3/Utility.hpp"

using namespace std::chrono_literals;

namespace netio3 {
    struct [[gnu::visibility("default")]] NetioPublisherConfig {
        /// Which netio3 backend to use
        NetworkType backend_type{NetworkType::LIBFABRIC};
        /// Network mode for backend, passed to backend
        NetworkMode backend_mode{NetworkMode::RDMA};
        /// Size of buffers for sending published data
        size_t buffersize{};
        /// Number of buffers for publishing data
        uint32_t nbuffers{};
        /// microseconds before an incomplete buffer is automatically flushed
        uint64_t flush_interval{};
        /// memory region start address for zero copy
        std::uint8_t* mr_start{nullptr};
        /// use buffered sending (multiple packets/buffer) or zero-copy
        SendMethod method{};
        /// Thread safety model for the backend
        ThreadSafetyModel thread_safety{ThreadSafetyModel::SAFE};
    };

    // publisher callbacks
    using CbSubscriptionReceived = std::function<void(uint64_t tag, const EndPointAddress& ep)>;
    using CbUnsubscriptionReceived = std::function<void(uint64_t tag, const EndPointAddress& ep)>;
    using CbResourceAvailable = std::function<void()>;
    using CbPublishCompleted = std::function<void(std::uint64_t key)>;

    enum class NetioPublisherStatus { OK, NO_RESOURCES, FAILED, NO_SUBSCRIPTIONS, PARTIALLY_FAILED };

    /// Publisher class. Publishes data to a set of subscribers by tag.
    /**
     * @brief The publisher class is responsible for sending data to a set of subscribers
     *
     * A publisher contains two senders and one receiver. The receiver is used to receive
     * subscription requests from subscribers. One of the senders then responds with an
     * acknoledgement. TCP using the ASYNCMSG provide is used for this communication. The other
     * sender is used to send the actual data to the subscribers. The connection type is determined
     * by the config passed to the constructor.
     *
     * Three different modes of sending data are supported:
     * - Buffered: Data is accumulated in a buffer and sent when the buffer is full or a timeout is
     * reached (see flush_interval).
     * - Send: Data is sent immediately using buffers (meaning the data is copied and the user can
     * discard it immediately).
     * - Zero-copy: Data is sent without copying it. The data must be in a registered memory region.
     * It must also remain valid until the data has been sent.
     *
     * Zero-copy mode requires to pass a key to track when all send operations beloning to the same
     * publish operation have completed. A callback set by set_on_publish_completed will be invoked
     * with the key.
     */
    template<SubscriptionEndpointContainer Container>
    class [[gnu::visibility("default")]] NetioPublisherTemplate {
    public:
        /**
         * @brief Constructor for publisher
         *
         * @param config     configuration of the network backend and buffering
         * @param evloop     pointer to event loop that runs netio operations
         * @param local_ep   Endpoint on which we listen for subscription requests
         */
        explicit NetioPublisherTemplate(const NetioPublisherConfig& config,
                                        std::shared_ptr<BaseEventLoop> eventloop,
                                        EndPointAddress local_ep);

        NetioPublisherTemplate(const NetioPublisherTemplate&) = delete;
        NetioPublisherTemplate(const NetioPublisherTemplate&&) = delete;
        NetioPublisherTemplate() = delete;
        NetioPublisherTemplate& operator=(const NetioPublisherTemplate&) = delete;
        NetioPublisherTemplate& operator=(const NetioPublisherTemplate&&) = delete;
        ~NetioPublisherTemplate() = default;

        /**
         * @brief Publish data to all subscribers of a tag
         *
         * If the retry parameter is true only the endpoints that failed on the previous attempt are
         * tried this time.
         *
         * @param tag    The tag being published
         * @param data   The data to publish
         * @param retry  Flag for first try at publish or a retry of failed subscriptions
         * @param user_status User supplied status code to be added to header
         *
         * @retval NetioPublisherStatus::OK Publish succeded to all subscibers
         * @retval NetioPublisherStatus::NO_RESOURCES publish failed to one or more subscribers due to a lack
         * of resources. A retry may succeed
         * @retval NetioPublisherStatus::FAILED publish to at least one subscriber failed but a retry will
         * not work
         * @retval NetioPublisherStatus::NO_SUBSCRIPTIONS There are currently no subscribers for this tag
         */
        [[nodiscard]] NetioPublisherStatus publish(uint64_t tag,
                                                   std::span<const iovec> data,
                                                   bool retry,
                                                   uint8_t user_status = 0) {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            return publish_internal(tag, data, retry, user_status);
        }

        /**
         * @overload
         */
        [[nodiscard]] NetioPublisherStatus publish(uint64_t tag,
                                                   std::span<const uint8_t> data,
                                                   bool retry,
                                                   uint8_t user_status = 0) {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            return publish_internal(tag, data, retry, user_status);
        }

        /**
         * @brief Publish data to all subscribers of a tag in zero-copy mode
         *
         * If the retry parameter is true only the endpoints that
         * failed on the previous attempt are tried this time.
         *
         * @param tag    The tag being published
         * @param data   The data to publish
         * @param retry  Flag for first try at publish or a retry of failed subscriptions
         * @param user_status User supplied status code to be added to header
         * @param key    Key for zero copy sending (to be returned once operation compelted)
         *
         * @retval NetioPublisherStatus::OK  Publish succeded to all subscibers
         * @retval NetioPublisherStatus::NO_RESOURCES publish failed to one or more subscribers
         * due to a lack of resources. A retry may succeed
         * @retval NetioPublisherStatus::FAILED publish to at least one subscriber failed and non
         * succeeded (retry will not work and no completion callback will be triggered)
         * @retval NetioPublisherStatus::PARTIALLY_FAILED publish to at least one subscriber
         * failed but a at least one send succeeded (retry will not work but a completion callback
         * wll be triggered)
         * @retval NetioPublisherStatus::NO_SUBSCRIPTIONS There are currently no subscribers for
         * this tag
         */
        [[nodiscard]] NetioPublisherStatus publish(uint64_t tag,
                                                   std::span<const iovec> data,
                                                   bool retry,
                                                   uint8_t user_status,
                                                   uint64_t key) {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            return publish_internal(tag, data, retry, user_status, key);
        }

        /**
         * @overload
         */
        [[nodiscard]] NetioPublisherStatus publish(uint64_t tag,
                                                   std::span<uint8_t> data,
                                                   bool retry,
                                                   uint8_t user_status,
                                                   uint64_t key) {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            return publish_internal(tag, data, retry, user_status, key);
        }

        /**
         * @brief Flush current buffer for all subscriber endpoints
         *
         * @return NO_RESOURCES if any sender returned it, FAILED if any
         * returned that, OK otherwise
         */
        [[nodiscard]] NetioPublisherStatus flush_buffer();

        /**
         * @brief Get the actual endpoint address the publisher is listening on
         *
         * This may be different to the endpoint given to the constructor if no port was specified
         * at that time.
         *
         * @return address of local endpoint
         */
        EndPointAddress get_endpoint() { return m_local_ep; };

        /**
         * @brief Get the current statistics for the publisher
         *
         * @return PublisherStats structure containing the current stats
         */
        [[nodiscard]] PublisherStats get_stats();

        /**
         * @brief Set a callback function to be called on subscription
         *
         * @param cb Callback function
         */
        void set_on_subscription(const CbSubscriptionReceived& cb);
        /**
         * @brief Set a callback function to be called on unsubscription
         *
         * @param cb Callback function
         */
        void set_on_unsubscription(const CbUnsubscriptionReceived& cb);

        /**
         * @brief Set a callback function to be called when an unspecified resource becomes avalable
         *
         * @param cb Callback function
         */
        void set_on_resource_available(const CbResourceAvailable& cb);

        /**
         * @brief Set a callback function to be called when data has been sent to all subscribers
         *
         * @param cb Callback function
         */
        void set_on_publish_completed(const CbPublishCompleted& cb);

    private:
        struct SubAcknowledgement {
            std::uint64_t tag{};
            SubscriptionRequest::Sub ack{};

            [[nodiscard]] constexpr auto operator<=>(const SubAcknowledgement&) const = default;
        };

        /**
         * @brief Handle closure of a connection of the data sender
         *
         * Invoked when a connection of the data sender was closed. Cleans up all subscriptions
         * belonging to the endpoint.
         *
         * @param ep The endpoint that closed
         * @param keys Keys for the send operations that were in flight
         */
        [[gnu::visibility("hidden")]]
        void sender_closed(const EndPointAddress& ep, const std::vector<std::uint64_t>& keys);

        /**
         * @brief Handle closure of a connection of the acknowledgement sender
         *
         * Drops any pending subscription acknowledgements for the endpoint.
         *
         * @param ep The endpoint that closed
         */
        [[gnu::visibility("hidden")]]
        void sender_ack_closed(const EndPointAddress& ep);

        /**
         * @brief Handle closure of a connection of the acknowledgement receiver
         *
         * Log incident and do nothing.
         *
         * @param ep The endpoint that closed
         */
        [[gnu::visibility("hidden")]]
        void receiver_closed(const EndPointAddress& ep);

        /**
         * @brief Handle a subscription request (either subscription or unsubscription)
         *
         * Either add the subscription to the list of subscriptions or remove it. If the last
         * subscription using a connection is removed, the connection is closed.
         *
         * @param data The data containing subscription request
         */
        [[gnu::visibility("hidden")]]
        void subscription_callback(std::span<const uint8_t> data);

        /**
         * @brief Handle a subscription request
         *
         * Open connections for both data and acknowledgement if necessary and add this endpoint to
         * the list of subscriptions. Send an acknowledgement and add it to the retry list if
         * sending failed with NO_RESOURCES (sender not yet open, or busy).
         *
         * @param data The data containing the acknowledgement
         */
        [[gnu::visibility("hidden")]]
        void add_subscription(const SubscriptionRequest& req, const EndPointAddress& ep);

        /**
         * @brief Handle an unsubscription request
         *
         * Remove the subscription from the list of subscriptions. Drop any failed sends for this
         * endpoint. Send an acknowledgement and add it to the retry list if sending failed with
         * NO_RESOURCES.
         */
        [[gnu::visibility("hidden")]]
        void remove_subscription(uint64_t tag, const EndPointAddress& ep);

        /**
         * @brief Add an (un)subscription acknowledgement to the retry list
         *
         * @param tag The tag of the subscription
         * @param ep The endpoint to send the acknowledgement to
         * @param sub The subscription request
         */
        [[gnu::visibility("hidden")]]
        void add_ack_to_retry_list(uint64_t tag, const EndPointAddress& ep, SubscriptionRequest::Sub sub);

        /**
         * @brief Retry sending pending (un)subscription acknowledgements
         *
         * Executed on a timer. Deletes them from the list if sending succeeds. Stops the timer if
         * the list is empty.
         */
        [[gnu::visibility("hidden")]]
        void retry_subscription_acks();

        /**
         * @brief Callback for completion of a send operation
         *
         * Invoke resource available callback if they were exhausted before. For zero-copy, check if
         * all sends for the same key have completed and invoke the user callback if so.
         */
        [[gnu::visibility("hidden")]]
        void send_complete_callback(const EndPointAddress& ep, uint64_t key);

        /**
         * @brief Send data to a single endpoint
         *
         * @tparam DataType Type of the data to be published (span, span const or vector of iovec)
         * @param tag The tag being sent
         * @param ep The endpoint to send to
         * @param data The data to send
         * @param user_status User supplied status code to be added to header
         * @param key Key for zero copy sending
         * @return NetioStatus::OK if the send was successful, NetioStatus::NO_RESOURCES if the send
         * failed due to lack of resources, or NetioStatus::FAILED if the send failed for another
         * reason
         */
        template<typename DataType>
        [[gnu::visibility("hidden")]]
        [[nodiscard]] NetioStatus send_data(uint64_t tag,
                                            NetioSender::Connection& con,
                                            const DataType& data,
                                            uint8_t user_status,
                                            std::optional<uint64_t> key = std::nullopt)
            requires std::same_as<DataType, std::span<uint8_t>> ||
                     std::same_as<DataType, std::span<const uint8_t>> ||
                     std::same_as<std::decay_t<DataType>, std::span<const iovec>>;

        /**
         * @brief Publish data to all subscribers of a tag
         *
         * If the retry parameter is true only the endpoints that failed on the previous attempt are
         * tried this time.
         *
         * @tparam DataType Type of the data to be published (span, span const or vector of iovec)
         * @param tag    The tag being published
         * @param data   Descriptor of the data
         * @param retry  Flag for first try at publish or a retry of failed subscriptions
         * @param user_status User supplied status code to be added to header
         *
         * @retval NetioPublisherStatus::OK  Publish succeded to all subscibers
         * @retval NetioPublisherStatus::NO_RESOURCES publish failed to one or more subscribers due to a lack
         * of resources. A retry may succede.
         * @retval NetioPublisherStatus::FAILED publish to at least one subscriber failed but a retry will
         * not work
         * @retval NetioPublisherStatus::NO_SUBSCRIPTIONS There are currently no subscribers for this tag
         */
        template<typename DataType>
        [[gnu::visibility("hidden")]]
        [[nodiscard]] NetioPublisherStatus publish_internal(uint64_t tag,
                                                            const DataType& data,
                                                            bool retry,
                                                            uint8_t user_status)
            requires std::same_as<DataType, std::span<const uint8_t>> ||
                     std::same_as<std::decay_t<DataType>, std::span<const iovec>>;

        /**
         * @brief Publish data to all subscribers of a tag
         *
         * If the retry parameter is true only the endpoints that failed on the previous attempt are
         * tried this time.
         *
         * @tparam DataType Type of the data to be published (span, span const or vector of iovec)
         * @param tag    The tag being published
         * @param data   Descriptor of the data
         * @param retry  Flag for first try at publish or a retry of failed subscriptions
         * @param user_status User supplied status code to be added to header
         * @param key    Key for zero copy sending
         *
         * @retval NetioPublisherStatus::OK  Publish succeded to all subscibers
         * @retval NetioPublisherStatus::NO_RESOURCES publish failed to one or more subscribers
         * due to a lack of resources. A retry may succeed
         * @retval NetioPublisherStatus::FAILED publish to at least one subscriber failed and non
         * succeeded (retry will not work and no completion callback will be triggered)
         * @retval NetioPublisherStatus::PARTIALLY_FAILED publish to at least one subscriber
         * failed but a at least one send succeeded (retry will not work but a completion callback
         * wll be triggered)
         * @retval NetioPublisherStatus::NO_SUBSCRIPTIONS There are currently no subscribers for
         * this tag
         */
        template<typename DataType>
        [[gnu::visibility("hidden")]]
        [[nodiscard]] NetioPublisherStatus publish_internal(uint64_t tag,
                                                            const DataType& data,
                                                            bool retry,
                                                            uint8_t user_status,
                                                            uint64_t key)
            requires std::same_as<DataType, std::span<uint8_t>> ||
                     std::same_as<std::decay_t<DataType>, std::span<const iovec>>;

        /**
         * @brief Handle completion of a send operation
         *
         * Update tracker and call user callback if all sends for the same key have completed. To be
         * called by completed sends or closed connections.
         */
        [[gnu::visibility("hidden")]]
        void handle_send_completion(std::uint64_t key);

        /**
         * @brief Get the local link id for a tag
         *
         * @param tag The tag to get the local id for
         * @return The local id
         */
        [[gnu::visibility("hidden")]]
        [[nodiscard]] std::size_t get_local_id(std::uint64_t tag) const;

        /**
         * @brief Swap the failed send lists
         *
         * Swap the failed send lists and clear the next one for next iteration.
         *
         * @param tag The tag to swap
         */
        [[gnu::visibility("hidden")]]
        void swap_failed(std::uint64_t tag)
        {
            std::swap(m_failed_sends.at(tag), m_failed_sends_next.at(tag));
            m_failed_sends_next.at(tag).clear();
        }

        CbSubscriptionReceived m_on_subscription = [](auto&&...) {};
        CbUnsubscriptionReceived m_on_unsubscription = [](auto&&...) {};
        CbResourceAvailable m_on_resource_available = [](auto&&...) {};

        CbPublishCompleted m_on_publish_completed{nullptr};

        struct ConnectionEndpointPair {
            std::reference_wrapper<NetioSender::Connection> con;
        };

        Container m_subscription_endpoints;
        std::map<EndPointAddress, NetioSender::Connection&> m_connections;
        std::map<EndPointAddress, std::set<uint64_t>> m_subscriptions;
        std::map<EndPointAddress, EndPointAddress> m_ack_address;
        std::unique_ptr<NetioSender> m_sender;
        std::unique_ptr<NetioReceiver> m_receiver;
        NetioSender m_ack_sender;
        std::map<uint64_t, std::vector<std::reference_wrapper<NetioSender::Connection>>> m_failed_sends;
        std::map<uint64_t, std::vector<std::reference_wrapper<NetioSender::Connection>>> m_failed_sends_next;
        EndPointAddress m_local_ep;
        std::map<EndPointAddress, std::set<SubAcknowledgement>> m_pending_subscription_acks;
        EventTimerHandle m_subscription_ack_retry_timer;
        std::atomic<std::size_t> m_num_subs{};
        std::map<uint64_t, int> m_pending_sends;
        std::uint64_t m_last_key{};
        bool m_track_pending_sends{};
        SendMethod m_method{};
        bool m_resources_exhausted;
        bool m_thread_safe{};
        std::recursive_mutex m_sub_mutex;
        std::mutex m_pending_sub_mutex;
        std::recursive_mutex m_sub_ep_mutex;
        std::recursive_mutex m_pending_send_mutex;

        static constexpr size_t c_recv_buffer_size = 0;
        static constexpr uint32_t c_nrecv_buffers = 0;
    };

    using NetioPublisher = NetioPublisherTemplate<SubscriptionEndpointMap>;

    template<SubscriptionEndpointContainer Container>
    NetioPublisherTemplate<Container>::NetioPublisherTemplate(
        const NetioPublisherConfig& config,
        std::shared_ptr<BaseEventLoop> evloop,
        EndPointAddress local_ep) :
        m_ack_sender{{
                       .backend_type = NetworkType::ASYNCMSG,
                       .backend_mode = NetworkMode::TCP,
                       .buffersize = c_recv_buffer_size,
                       .nbuffers = c_nrecv_buffers,
                       .flush_interval = 0,
                     },
                     evloop},
        m_local_ep(std::move(local_ep)),
        m_subscription_ack_retry_timer{
          evloop->create_timer([this](int) { retry_subscription_acks(); })},
        m_method(config.method),
        m_resources_exhausted(false),
        m_thread_safe{config.thread_safety == ThreadSafetyModel::SAFE} {
        ERS_DEBUG(5, "Entered");

        NetioReceiverConfig rcvr_config {
            .backend_type = NetworkType::ASYNCMSG,
            .backend_mode = NetworkMode::TCP,
        };

        m_receiver = std::make_unique<NetioReceiver>(rcvr_config, evloop);
        m_receiver->set_on_data_cb ([this](std::uint64_t, std::span<const std::uint8_t> data, std::uint8_t) {
            subscription_callback(data);
        });
        m_receiver->set_on_connection_closed_cb(
        [this](const EndPointAddress& ep) {
            receiver_closed(ep);
        });
        const auto port = m_receiver->listen(m_local_ep, {c_recv_buffer_size, c_nrecv_buffers});
        m_local_ep.port(port); // Update port in local endpoint to the actual port we're listening on

        NetioSenderConfig sender_config{
            .backend_type = config.backend_type,
            .backend_mode = config.backend_mode,
            .buffersize = config.buffersize,
            .nbuffers = config.nbuffers,
            .flush_interval = config.flush_interval,
            .mr_start = config.mr_start,
            .thread_safety = config.thread_safety,
        };

        m_sender = std::make_unique<NetioSender>(sender_config, evloop);
        m_sender->set_on_connection_closed_internal([this](const EndPointAddress& ep, const std::vector<std::uint64_t>& keys) {
            sender_closed(ep, keys);});
        m_sender->set_on_send_completed ([this](const EndPointAddress& ep,
                                                uint64_t key){
            send_complete_callback(ep, key);});

        m_ack_sender.set_on_connection_closed([this](const EndPointAddress& ep) {
            sender_ack_closed(ep);
        });

        ERS_DEBUG(5, "Finished");
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::sender_closed(const EndPointAddress& ep, const std::vector<std::uint64_t>& keys) {
        ERS_DEBUG(5, "Entered ep=" << ep);
        {
            // OK, now we have to erase all the subscriptions that are sending
            // to this endpoint
            utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
            if (m_subscriptions.contains(ep)) {
                ERS_DEBUG(5, m_subscriptions.at(ep).size()
                        << " active subscriptions using ep");
                auto subscriptions = m_subscriptions.at(ep);
                for (const auto& tag : subscriptions) {
                    remove_subscription(tag, ep);
                }
                ERS_DEBUG(5, "Now "<< m_subscriptions.at(ep).size()
                        << " active subscriptions using ep");
                m_subscriptions.erase(ep);
                m_connections.erase(ep);
            }
        }

        // Mark pending publish operations as finished
        if (m_method == SendMethod::ZERO_COPY) {
            utility::ConditionalLockGuard lock{m_pending_send_mutex, m_thread_safe};
            for (auto key : keys) {
                handle_send_completion(key);
            }
        }
        ERS_DEBUG(5, "Finished");
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::sender_ack_closed(const EndPointAddress& ep) {
        ERS_DEBUG(5, "Entered ep=" << ep);
        const utility::ConditionalLockGuard lock(m_pending_sub_mutex, m_thread_safe);
        const utility::ConditionalLockGuard lock2(m_sub_mutex, m_thread_safe);
        std::erase_if(m_ack_address, [&ep](const auto& pair) {
            return pair.second == ep;
        });
        m_pending_subscription_acks.erase(ep);
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::receiver_closed([[maybe_unused]] const EndPointAddress& ep) {
        ERS_DEBUG(5, "Entered ep=" << ep);

    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::send_complete_callback(const EndPointAddress& /* ep */,
                                                                   uint64_t key) {
        ERS_DEBUG(6, "key=" << key);
        if (m_resources_exhausted) {
            // We must have some resources since a send has completed
            m_resources_exhausted = false;
            // Tell the user
            ERS_DEBUG(6, "Calling resource available callback");
            m_on_resource_available();
        }
        if (m_method == SendMethod::ZERO_COPY) {
            utility::ConditionalLockGuard lock{m_pending_send_mutex, m_thread_safe};
            handle_send_completion(key);
        }
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::remove_subscription(uint64_t tag,
                                                                const EndPointAddress& ep) {
        // Remove from our subscription list
        {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            ERS_DEBUG(5, "Removing ep " << ep
                << " from subscriptions for tag 0x" << std::hex << tag << std::dec
                    << " tag currently has " << m_subscription_endpoints.at(tag).size()
                << " endpoints subscribed");
            if (not m_subscription_endpoints.contains(tag)) {
                ERS_DEBUG(5, std::format("Subscription for tag {:#x} not found, probably already removed. Do nothing", tag));
                return;
            }
            std::erase_if(m_subscription_endpoints.at(tag), [&ep](const auto& element) {
                return element.get().get_endpoint() == ep;
            });

            if (m_failed_sends.contains(tag)) {
                const auto entry = std::ranges::find_if(m_failed_sends.at(tag), [&ep](const auto& element) {
                    return element.get().get_endpoint() == ep;
                });
                if (entry != m_failed_sends.at(tag).end()) {
                    handle_send_completion(m_last_key);
                    m_failed_sends.at(tag).erase(entry);
                }
            }

            if (m_failed_sends_next.contains(tag)) {
                std::erase_if(m_failed_sends_next.at(tag), [&ep](const auto& element) {
                    return element.get().get_endpoint() == ep;
                });
            }

            if (m_subscription_endpoints.at(tag).empty()) {
                ERS_DEBUG(5, "No more subscriptions for tag 0x" << std::hex << tag << std::dec << " removing tag from map");
                m_subscription_endpoints.erase(tag);
                m_failed_sends.erase(tag);
                m_failed_sends_next.erase(tag);
            }
        }

        utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
        ERS_DEBUG(5, "Current subscriptions:  m_subscriptions.size()=" << m_subscriptions.size()
            << ",  m_subscriptions.at(ep).size()=" << m_subscriptions.at(ep).size());
        // Find the original subscription and remove it from our index
        if (m_subscriptions.contains(ep) and m_subscriptions.at(ep).contains(tag)) {
            if (m_ack_address.contains(ep)) {
                // Acknowledge the unsubscription
                // Note this is only allowed for ASYNCMSG as it copies the data. Libfabric would have a dangling pointer
                auto ack = SubscriptionRequest::Sub::UNSUBSCRIBE;
                const auto ret = m_ack_sender.send_data(
                    m_ack_address.at(ep),
                    tag,
                    std::span<uint8_t>(reinterpret_cast<uint8_t*>(&ack),
                                    sizeof(ack)),
                    0
                );
                if (ret == NetioStatus::NO_RESOURCES) {
                    add_ack_to_retry_list(tag, m_ack_address.at(ep), SubscriptionRequest::Sub::UNSUBSCRIBE);
                }
                else if (ret != NetioStatus::OK) {
                    ers::error(FailedAcknowledge(ERS_HERE, "unsubscription", tag));
                }
            }
            m_subscriptions.at(ep).erase(tag);
            m_num_subs.fetch_sub(1, std::memory_order_relaxed);
        }
        ERS_DEBUG(5, "Finished");
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::add_ack_to_retry_list(uint64_t tag, const EndPointAddress& ep, SubscriptionRequest::Sub sub) {
        utility::ConditionalLockGuard lock_pending_sub(m_pending_sub_mutex, m_thread_safe);
        if (not m_pending_subscription_acks.contains(ep)) {
            m_pending_subscription_acks.try_emplace(ep);
        }
        m_pending_subscription_acks.at(ep).emplace(tag, sub);
        if (not m_subscription_ack_retry_timer.is_running()) {
            m_subscription_ack_retry_timer.start(100ms);
        }
    }


    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::retry_subscription_acks() {
        ERS_DEBUG(3, "Retrying to send open subscription acknowledgements");
        utility::ConditionalLockGuard lock(m_pending_sub_mutex, m_thread_safe);
        for (auto& [ep, reqs] : m_pending_subscription_acks) {
            std::set<SubAcknowledgement> successes;
            // Note this is only allowed for ASYNCMSG as it copies the data. Libfabric would have a dangling pointer
            for (auto req : reqs) {
                ERS_DEBUG(3, "Sending pending ack for subscription to ep " << ep << std::format(" tag={:#16x}", req.tag));
                const auto ret = m_ack_sender.send_data(
                    ep,
                    req.tag,
                    {reinterpret_cast<uint8_t*>(&req.ack), sizeof(req.ack)},
                    0
                );
                ERS_DEBUG(3, "Sent pending ack for subscription to ep " << ep << " ret=" << static_cast<uint32_t>(ret) << std::format(" tag={:#16x}", req.tag));
                if (ret == NetioStatus::OK) {
                    successes.emplace(req);
                }
            }
            std::erase_if(m_pending_subscription_acks.at(ep), [&successes](const auto& element) {
                return successes.contains(element);
            });
        }
        std::erase_if(m_pending_subscription_acks, [](const auto& pair) {
            return pair.second.empty();
        });
        if (m_pending_subscription_acks.empty()) {
            m_subscription_ack_retry_timer.stop();
        }
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::add_subscription(const SubscriptionRequest& req,
                                                             const EndPointAddress& ep) {
        ERS_DEBUG(5, "Subscription request for tag 0x" << std::hex << req.tag
            << " (" << std::dec << req.tag << ")"
            << " from ep=" << ep);
        // Add to our subscription list
        {
            utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
            if (!m_subscriptions.contains(ep)) {
                if (req.ack_port != 0) {
                    const EndPointAddress ackEp{ep.address(), req.ack_port};
                    ERS_DEBUG(5, "Opening connection for acknowledgements: ep=" << ackEp);
                    m_ack_sender.open_connection(ackEp);
                    m_ack_address.emplace(std::pair(ep,ackEp));
                }
                auto& con = m_sender->open_connection(ep);
                m_subscriptions.emplace(std::pair(ep, std::set<uint64_t>()));
                m_connections.try_emplace(ep, con);
            }
            m_subscriptions.at(ep).insert(req.tag);
        }
        m_num_subs.fetch_add(1, std::memory_order_relaxed);

        {
            utility::ConditionalLockGuard lock{m_sub_ep_mutex, m_thread_safe};
            if (not m_subscription_endpoints.contains(req.tag)) {
                m_subscription_endpoints.add(req.tag);
            }
            if (std::ranges::find_if(m_subscription_endpoints.at(req.tag), [&ep](const auto& entry) {
                    return entry.get().get_endpoint() == ep;
                }) == m_subscription_endpoints.at(req.tag).end()) {
                m_subscription_endpoints.at(req.tag).emplace_back(m_connections.at(ep));
            }
            if (!m_failed_sends.contains(req.tag)) {
                m_failed_sends.try_emplace(req.tag, std::vector<std::reference_wrapper<NetioSender::Connection>>{});
            }
            m_failed_sends.at(req.tag).reserve(m_subscription_endpoints.at(req.tag).size());
            if (!m_failed_sends_next.contains(req.tag)) {
                m_failed_sends_next.try_emplace(req.tag, std::vector<std::reference_wrapper<NetioSender::Connection>>{});
            }
            m_failed_sends_next.at(req.tag).reserve(m_subscription_endpoints.at(req.tag).size());
        }
        // then call user's on_subscription callback
        m_on_subscription(req.tag, ep);

        if (m_ack_address.contains(ep)) {
            // Opening takes too long for us to do the ack here!
            // Should have it done in the connection established callback!!!
            // Maybe have a vector of pending acks that we send in the callback?
            auto& ackEp = m_ack_address.at(ep);
            ERS_DEBUG(5, "Sending ack for subscription to ep " << ep
                    << " to ack ep " << ackEp);
            // Note this is only allowed for ASYNCMSG as it copies the data. Libfabric would have a dangling pointer
            auto ack = SubscriptionRequest::Sub::SUBSCRIBE;
            const auto ret = m_ack_sender.send_data(
                m_ack_address.at(ep),
                req.tag,
                {reinterpret_cast<uint8_t*>(&ack), sizeof(ack)},
                0
            );
            if (ret == NetioStatus::NO_RESOURCES) {
                add_ack_to_retry_list(req.tag, ackEp, SubscriptionRequest::Sub::SUBSCRIBE);
            }
            else if (ret != NetioStatus::OK) {
                ers::error(FailedAcknowledge(ERS_HERE, "subscription", req.tag));
            }
        }

        utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
        ERS_DEBUG(6, "Current subscriptions:  m_subscriptions.size()=" << m_subscriptions.size()
                << ",  m_subscriptions.at(ep).size()=" << m_subscriptions.at(ep).size());
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::subscription_callback(std::span<const uint8_t> data) {
        SubscriptionRequest req;
        try {
            req = SubscriptionRequest::from_json(std::string(data.begin(), data.end()));
        } catch (const SubscriptionParseError& ex) {
            ers::error(ex);
            return;
        }
        ERS_DEBUG(7, "SubscriptionRequest=" << SubscriptionRequest::to_json(req));

        const EndPointAddress ep(req.addr, req.data_port);
        if (req.subscribe == SubscriptionRequest::Sub::SUBSCRIBE) {
            add_subscription(req, ep);
        }
        else {
            ERS_DEBUG(5, "Unsubscribe request for tag 0x"
                    << std::hex << req.tag << std::dec << " from ep=" << ep);

            remove_subscription(req.tag, ep);

            // We should close the end point if it is not needed by any of our
            // other subscriptions
            {
                utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
                if (not m_subscriptions.contains(ep)) {
                    ERS_DEBUG(5, std::format("Subscription for endpoint {}:{} not found, probably already removed. Do nothing", ep.address(), ep.port()));
                    return;
                }
                if (m_subscriptions.at(ep).empty()) {
                    ERS_DEBUG(5, "No more subscriptions to ep " << ep
                            << " closing connection");
                    m_sender->close_connection(ep);
                    m_subscriptions.erase(ep);
                    m_connections.erase(ep);
                    if (m_ack_address.contains(ep)) {
                        m_ack_sender.close_connection(m_ack_address.at(ep));
                        m_ack_address.erase(ep);
                    }
                }
            }
            // then call user's on_unsubscription callback
            m_on_unsubscription(req.tag, ep);
            utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
            ERS_DEBUG(5, "Remaining subscriptions:  m_subscriptions.size()=" << m_subscriptions.size());
            if (m_subscriptions.contains(ep)) {
                ERS_DEBUG(5, "m_subscriptions.at(ep).size()=" << m_subscriptions.at(ep).size());
            }
            [[maybe_unused]] int snum = 0;
            for ([[maybe_unused]] const auto& subs : m_subscriptions) {
                ERS_DEBUG(5, "subs " << ++snum << " of " << m_subscriptions.size() << " subs.first=" << subs.first);
            }
            ERS_DEBUG(5, "Remaining subscriptions:  m_subscriptions.size()=" << m_subscriptions.size());
        }
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::handle_send_completion(std::uint64_t key)
    {
        if (not m_pending_sends.contains(key)) {
            return;
        }
        --m_pending_sends.at(key);
        if (m_pending_sends.at(key) == 0) {
            m_pending_sends.erase(key);
            if (m_on_publish_completed) {
                m_on_publish_completed(key);
            }
        }
    }

    template<SubscriptionEndpointContainer Container>
    std::size_t NetioPublisherTemplate<Container>::get_local_id(std::uint64_t tag) const
    {
        return (tag & 0xFFF0000ULL) >> 16;
    }

    template<SubscriptionEndpointContainer Container>
    template<typename DataType>
    NetioStatus NetioPublisherTemplate<Container>::send_data(
      uint64_t tag,
      NetioSender::Connection& con,
      const DataType& data,
      uint8_t user_status,
      std::optional<std::uint64_t> key)
        requires std::same_as<DataType, std::span<uint8_t>> ||
                 std::same_as<DataType, std::span<const uint8_t>> ||
                 std::same_as<std::decay_t<DataType>, std::span<const iovec>>
    {
        ERS_DEBUG(10, "Sending");
        switch (m_method) {
            case SendMethod::BUFFERED:
                return m_sender->buffered_send_data(con, tag, data, user_status);
            case SendMethod::ZERO_COPY:
                if constexpr (std::same_as<DataType, std::span<const uint8_t>>) {
                    throw std::invalid_argument("Zero copy requires non-const data");
                } else {
                    if (key.has_value()) {
                        return m_sender->zero_copy_send_data(con, tag, data, user_status, key.value());
                    }
                }
                throw std::invalid_argument("Zero copy requires calling publish with key");
            case SendMethod::SEND:
                return m_sender->send_data(con, tag, data, user_status);
            default:
                throw std::invalid_argument("Invalid send method");
        }
    }

    template<SubscriptionEndpointContainer Container>
    template<typename DataType>
    NetioPublisherStatus NetioPublisherTemplate<Container>::publish_internal(uint64_t tag,
                                                                             const DataType& data,
                                                                             bool retry,
                                                                             uint8_t user_status)
        requires std::same_as<DataType, std::span<const uint8_t>> ||
                 std::same_as<std::decay_t<DataType>, std::span<const iovec>>
    {
        if (m_method == SendMethod::ZERO_COPY) [[unlikely]] {
            ers::error(WrongPublishMethodCalled("Called zero-copy publish function when not configured to use zero-copy"));
            return NetioPublisherStatus::FAILED;
        }

        // Lookup tag in subscription list and send to appropriate end points
        if (not m_subscription_endpoints.contains(tag)) [[unlikely]] {
            ERS_DEBUG(6, "No subscriptions for tag "
                << std::hex << tag << std::dec);
            return NetioPublisherStatus::NO_SUBSCRIPTIONS;
        }

        NetioPublisherStatus status = NetioPublisherStatus::OK;
        bool try_again = false;
        const auto& connections = std::invoke([this, &retry, &tag] () -> const auto& {
            if (retry) [[unlikely]] {
                return m_failed_sends.at(tag);
            }
            return m_subscription_endpoints.at(tag);
        });
        for (const auto& con : connections) {
            ERS_DEBUG(6, "Publishing tag 0x" << std::hex << tag << std::dec
                      << " to " << con.get().get_endpoint() << " retry=" << retry);
            try {
                auto ep_status = send_data(tag, con, data, user_status);
                if (ep_status == NetioStatus::FAILED) [[unlikely]] {
                    m_failed_sends_next.at(tag).clear();
                    return NetioPublisherStatus::FAILED;
                }
                if (ep_status == NetioStatus::NO_RESOURCES) [[unlikely]] {
                    m_failed_sends_next.at(tag).emplace_back(con);
                    m_resources_exhausted = true;
                    try_again = true;
                }
            }
            catch (const std::runtime_error& exception) {
                ers::error(FailedSend(ERS_HERE, con.get().get_endpoint().address(), con.get().get_endpoint().port(),
                                    exception.what(), exception));
                m_failed_sends_next.at(tag).clear();
                return NetioPublisherStatus::FAILED;
            }
        }
        if (try_again or retry) [[unlikely]] {
            swap_failed(tag);
        }
        if (try_again) [[unlikely]] {
            return NetioPublisherStatus::NO_RESOURCES;
        }
        return status;
    }

    template<SubscriptionEndpointContainer Container>
    template<typename DataType>
    NetioPublisherStatus NetioPublisherTemplate<Container>::publish_internal(uint64_t tag,
                                                                             const DataType& data,
                                                                             bool retry,
                                                                             uint8_t user_status,
                                                                             std::uint64_t key)
        requires std::same_as<DataType, std::span<uint8_t>> ||
                 std::same_as<std::decay_t<DataType>, std::span<const iovec>>
    {
        if (m_method != SendMethod::ZERO_COPY) [[unlikely]] {
            ers::error(WrongPublishMethodCalled("Called zero-copy publish function when not configured to use zero-copy"));
            return NetioPublisherStatus::FAILED;
        }

        // Lookup tag in subscription list and send to appropriate end points
        if (not m_subscription_endpoints.contains(tag)) {
            ERS_DEBUG(6, "No subscriptions for tag "
                << std::hex << tag << std::dec);
            return NetioPublisherStatus::NO_SUBSCRIPTIONS;
        }

        NetioPublisherStatus status = NetioPublisherStatus::OK;
        bool try_again = false;
        bool none_succeded = true;
        int counter_ok = 0;
        int counter_again = 0;
        const auto& connections = std::invoke([this, &retry, &tag] () -> const auto& {
            if (retry) [[unlikely]] {
                return m_failed_sends.at(tag);
            }
            return m_subscription_endpoints.at(tag);
        });
        for (const auto& con : connections) {
            ERS_DEBUG(6, "Publishing tag 0x" << std::hex << tag << std::dec
                    << " to " << con.get().get_endpoint() << " retry=" << retry);
            try {
                auto ep_status = send_data(tag, con, data, user_status, key);
                if (ep_status == NetioStatus::OK) {
                    none_succeded = false;
                    ++counter_ok;
                }
                if (ep_status == NetioStatus::FAILED) [[unlikely]] {
                    status = NetioPublisherStatus::FAILED;
                    m_failed_sends_next.clear();
                    break;
                }
                if (ep_status == NetioStatus::NO_RESOURCES) [[unlikely]] {
                    m_failed_sends_next.at(tag).emplace_back(con);
                    m_resources_exhausted = true;
                    try_again = true;
                    ++counter_again;
                }
            }
            catch (const std::runtime_error& exception) {
                ers::error(FailedSend(ERS_HERE, con.get().get_endpoint().address(), con.get().get_endpoint().port(),
                                    exception.what(), exception));
                status = NetioPublisherStatus::FAILED;
                m_failed_sends_next.clear();
                break;
            }
        }
        m_last_key = key;
        if (status == NetioPublisherStatus::FAILED and not none_succeded) [[unlikely]] {
            status = NetioPublisherStatus::PARTIALLY_FAILED;
        }
        else if (try_again) [[unlikely]] {
            status = NetioPublisherStatus::NO_RESOURCES;
        }
        {
            utility::ConditionalLockGuard lock{m_pending_send_mutex, m_thread_safe};
            if (not retry) {
                if (status == NetioPublisherStatus::OK or status == NetioPublisherStatus::NO_RESOURCES) {
                    m_pending_sends[key] += counter_ok + counter_again;
                } else {
                    m_pending_sends[key] += counter_ok;
                }
            } else {
                if (status == NetioPublisherStatus::FAILED or status == NetioPublisherStatus::PARTIALLY_FAILED) {
                    m_pending_sends[key] -= (m_failed_sends.at(tag).size() - counter_ok);
                }
            }
        }
        if (try_again or retry) [[unlikely]] {
            swap_failed(tag);
        }
        return status;
    }

    template<SubscriptionEndpointContainer Container>
    NetioPublisherStatus NetioPublisherTemplate<Container>::flush_buffer() {
        utility::ConditionalLockGuard lock{m_sub_mutex, m_thread_safe};
        std::vector<NetioStatus> statuses;
        statuses.reserve(m_subscriptions.size());
        std::ranges::transform(m_subscriptions, std::back_inserter(statuses),
                            [this](const auto& pair) {
                                return m_sender->flush_buffer(pair.first);
                            });
        if (std::ranges::any_of(statuses, [](const auto& status) {
                return status == NetioStatus::NO_RESOURCES;
            })) {
            return NetioPublisherStatus::NO_RESOURCES;
        }
        if (std::ranges::any_of(statuses, [](const auto& status) {
                return status == NetioStatus::FAILED;
            })) {
            return NetioPublisherStatus::FAILED;
        }
        return NetioPublisherStatus::OK;
    }

    template<SubscriptionEndpointContainer Container>
    netio3::PublisherStats NetioPublisherTemplate<Container>::get_stats() {
        return {
            .buffer_stats = m_sender->get_num_available_buffers(),
            .num_subscriptions = m_num_subs.load(),
        };
    }

    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::set_on_subscription(const CbSubscriptionReceived& cb) {
        m_on_subscription = cb;
    }
    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::set_on_unsubscription(const CbUnsubscriptionReceived& cb) {
        m_on_unsubscription = cb;
    }
    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::set_on_resource_available(const CbResourceAvailable& cb) {
        m_on_resource_available = cb;
    }
    template<SubscriptionEndpointContainer Container>
    void NetioPublisherTemplate<Container>::set_on_publish_completed(const CbPublishCompleted& cb) {
        if (m_method != SendMethod::ZERO_COPY) {
            ers::warning(OnPublishCallbackNotAvailable());
        }
        m_on_publish_completed = cb;
    }

}  // namespace netio3

#endif  // NETIO3_NETIOPUBLISHER_HPP
