#ifndef NETIO3_NETIORECEIVER_HPP
#define NETIO3_NETIORECEIVER_HPP

#include <memory>
#include <cstdint>
#include <functional>
#include <set>
#include <span>

#include <netio3-backend/EventLoop/BaseEventLoop.hpp>
#include <netio3-backend/Netio3Backend.hpp>

#include "netio3/Types.hpp"

namespace netio3 {
    using CbMessageReceived = std::function<void(std::uint64_t tag, std::span<const std::uint8_t> data, std::uint8_t status)>;
    using CbBufferReceived = std::function<void(std::span<const uint8_t> buffer)>;

    struct [[gnu::visibility("default")]] NetioReceiverConfig {
        NetworkType backend_type{NetworkType::LIBFABRIC};
        NetworkMode backend_mode{NetworkMode::RDMA};
        ThreadSafetyModel thread_safety{ThreadSafetyModel::SAFE};
    };

    /**
     * @brief The receiver class is responsible for receiving data from a sender
     *
     * The receiver also needs to know the sender mode of the publisher to know how to decode the
     * data.
     *
     * On data reception, the receiver will call the on_data_cb callback with the received decoded
     * data and/or the on_buffer_cb callback with the received buffer.
     */
    class [[gnu::visibility("default")]] NetioReceiver {
    public:
        /**
         * @brief Constructor for the NetioReceiver
         *
         * @param config Configuration of the network backend and buffers
         * @param evloop Pointer to the event loop that schedules netio ops
         */
        [[gnu::visibility("default")]]
        explicit NetioReceiver(const NetioReceiverConfig& config,
                               std::shared_ptr<BaseEventLoop> evloop);
        NetioReceiver(const NetioReceiver&) = delete;
        NetioReceiver(const NetioReceiver&&) = delete;
        NetioReceiver() = delete;
        NetioReceiver& operator=(const NetioReceiver&) = delete;
        NetioReceiver& operator=(const NetioReceiver&&) = delete;

        [[gnu::visibility("default")]]
        ~NetioReceiver();

        /**
         * @brief Listen for incoming connections
         *
         * @param ep Local endpoint address to listen on
         * @param conn_params Configuration for the opened receive endpoints
         *
         * @return The port of which we are listening (assigned by the system if 0 was passed)
         */
        [[gnu::visibility("default")]]
        unsigned short listen(const EndPointAddress& ep,
                              const ConnectionParametersRecv& conn_params);

        /**
         * @brief Close a listening endpoint
         *
         * Stop listening on the provided endpoint. Also closes all receive endpoints spawned by
         * this listener.
         *
         * @param ep Address of the endpoint
         */
        [[gnu::visibility("default")]]
        void close(const EndPointAddress& ep);

        /**
         * @brief Set a callback function to be called when data are received
         *
         * This callback is the main purpose of this class. Invoked for each decoded message inside
         * a buffer.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_data_cb(CbMessageReceived cb);

        /**
         * @brief Set a callback function to be called when data are received
         *
         * This callback is the main purpose of this class. Invoked with the entire buffer.
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_buffer_cb(CbBufferReceived cb);

        /**
         * @brief Set a callback function to be called when a connection is established
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_connection_established_cb(OnConnectionEstablishedSimpleCb cb);

        /**
         * @brief Set a callback function to be called when a connection is closed
         *
         * @param cb Callback function
         */
        [[gnu::visibility("default")]]
        void set_on_connection_closed_cb(OnConnectionClosedCb cb);

    private:
        NetworkType m_backend_type{};
        std::set<EndPointAddress> m_listeners;
        std::unique_ptr<NetworkBackend> m_backend;
        CbMessageReceived m_on_data_cb{nullptr};
        CbBufferReceived m_on_buffer_cb{nullptr};
        OnConnectionEstablishedSimpleCb m_on_connection_established_cb = [](auto&&...) {};
        OnConnectionClosedCb m_on_connection_closed_cb = [](auto&&...) {};

        /**
         * @brief Callback for when data are received
         *
         * Decodes the data and calls the user callback(s).
         *
         * @param data The received data
         */
        void on_data(std::span<const uint8_t> data);

        /**
         * @brief Callback for when a connection is established
         *
         * Call the user callback.
         *
         * @param ep The endpoint that was established
         */
        void connection_established(const EndPointAddress& ep);

        /**
         * @brief Callback for when a connection is closed
         *
         * Call the user callback.
         *
         * @param ep The endpoint that was closed
         */
        void connection_closed(const EndPointAddress& ep);
    };
}  // namespace netio3

#endif  // NETIO3_NETIORECEIVER_HPP