#ifndef NETIO3_UTILITY_HPP
#define NETIO3_UTILITY_HPP

namespace netio3::utility {
    /**
     * @brief Conditional lock guard
     *
     * Locks a mutex if a condition is true, and unlocks it when destroyed.
     *
     * @tparam Mutex The mutex type
     */
    template<typename Mutex>
    class ConditionalLockGuard {
    public:
        /**
         * @brief Construct a new Conditional Lock Guard object
         *
         * Lock the mutex if should_lock is true.
         *
         * @param mtx The mutex to lock
         * @param should_lock Whether to lock the mutex
         */
        ConditionalLockGuard(Mutex& mtx, bool should_lock) :
          m_mtx(mtx), m_should_lock(should_lock) {
            if (m_should_lock) {
                m_mtx.lock();
            }
        }

        /**
         * @brief Destroy the Conditional Lock Guard object
         *
         * Unlock the mutex if it was locked
         */
        ~ConditionalLockGuard() {
            if (m_should_lock) {
                m_mtx.unlock();
            }
        }

        ConditionalLockGuard(const ConditionalLockGuard&) = delete;
        ConditionalLockGuard& operator=(const ConditionalLockGuard&) = delete;
        ConditionalLockGuard(ConditionalLockGuard&&) = delete;
        ConditionalLockGuard& operator=(ConditionalLockGuard&&) = delete;

    private:
        Mutex& m_mtx;
        bool m_should_lock{};
    };

    /**
     * @brief Conditional unique lock
     *
     * Locks a mutex if a condition is true, and unlocks it when destroyed.
     *
     * @tparam Mutex The mutex type
     */
    template<typename Mutex>
    class ConditionalUniqueLock {
    public:
        /**
         * @brief Construct a new Conditional Lock Guard object
         *
         * Lock the mutex if should_lock is true.
         *
         * @param mtx The mutex to lock
         * @param should_lock Whether to lock the mutex
         */
        ConditionalUniqueLock(Mutex& mtx, bool should_lock) :
          m_mtx(mtx), m_should_lock(should_lock) {
            if (m_should_lock) {
                m_mtx.lock();
            }
        }

        /**
         * @brief Destroy the Conditional Lock Guard object
         *
         * Unlock the mutex if it was locked
         */
        ~ConditionalUniqueLock() {
            if (m_should_lock) {
                m_mtx.unlock();
            }
        }

        /**
         * @brief Lock the mutex
         */
        void lock() {
            if (!m_should_lock) {
                m_mtx.lock();
            }
        }

        /**
         * @brief Unlock the mutex
         */
        void unlock() {
            if (!m_should_lock) {
                m_mtx.unlock();
            }
        }

        ConditionalUniqueLock(const ConditionalUniqueLock&) = delete;
        ConditionalUniqueLock& operator=(const ConditionalUniqueLock&) = delete;
        ConditionalUniqueLock(ConditionalUniqueLock&&) = delete;
        ConditionalUniqueLock& operator=(ConditionalUniqueLock&&) = delete;

    private:
        Mutex& m_mtx;
        bool m_should_lock{};
    };
}  // namespace netio3::utility

#endif  // NETIO3_UTILITY_HPP