from enum import Enum
from typing import Annotated
# Please use `typing_extensions.TypedDict` instead of `typing.TypedDict` on Python < 3.12.
from typing_extensions import TypedDict

from fastapi import APIRouter, Depends

from felix_io_api.api import BitFieldRecord, DeviceRecord, Tags, get_shared


class Datatables(Enum):
    data = "data"


# for JavaScript, string only, no ints
class BitFieldRow(TypedDict):
    desc: str
    address: str
    mask: str
    bf_hi: str
    bf_lo: str
    fw_modes: str
    type: str  # noqa: A003
    endpoints: str
    alias: str
    group: str
    convert: str
    power: str
    multiply: str
    divide: str
    offset: str
    clear: str
    name: str
    value: str
    unit: str
    raw_value: str
    decoded_value: str


#
# Data
#
data_router = APIRouter()


# Converts integers to strings as JavaScript has a limitation of 2^53
def record_to_row(records: dict[str, BitFieldRecord]) -> list[BitFieldRow]:
    data = []
    for name, value in records.items():
        info = value['info']
        row: BitFieldRow = value.copy()
        row['desc'] = info['desc']
        row['address'] = str(info['address'])
        row['mask'] = str(info['mask'])
        row['bf_hi'] = str(info['bf_hi'])
        row['bf_lo'] = str(info['bf_lo'])
        row['fw_modes'] = info['fw_modes']
        row['type'] = info['type']
        row['endpoints'] = info['endpoints']
        row['alias'] = info.get('alias', '')
        row['group'] = info.get('group', '')
        row['convert'] = info.get('convert', '')
        row['power'] = str(info.get('power', ''))
        row['multiply'] = str(info.get('multiply', ''))
        row['divide'] = str(info.get('divide', ''))
        row['offset'] = str(info.get('offset', ''))
        row['clear'] = str(info.get('clear', ''))
        row['name'] = name
        row['unit'] = value.get('unit', '')
        # NOTE quickly convert array in decoded_value into ordinal html list
        if isinstance(row['decoded_value'], list):
            row['value'] = f"[{',<br/>'.join(str(x) for x in value['value'])}]"
            row['raw_value'] = f"[{',<br/>'.join(str(x) for x in value.get('raw_value', value['value']))}]"
            row['decoded_value'] = f"<ol start=\"0\">{''.join(f'<li>{x}</li>' for x in value['decoded_value'])}</ol>"
        else:
            row['value'] = str(value['value'])
            row['raw_value'] = str(value.get('raw_value', value['value']))
            row['decoded_value'] = value['decoded_value']

        data.append(row)

    return data


@data_router.get("/data/device/{device_id}/bitfield/{name}", tags=[Tags.data])
async def data_device_bitfield_name(shared: Annotated[dict, Depends(get_shared)],
                                    device_id:  int,
                                    name: str
                                    ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_bitfield(name, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/bitfield", tags=[Tags.data])
async def data_device_bitfield(shared: Annotated[dict, Depends(get_shared)],
                               device_id:  int
                               ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_list(device.bitfield_names(), info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/i2c/{name}", tags=[Tags.data])
async def data_device_i2c_name(shared: Annotated[dict, Depends(get_shared)],
                               device_id:  int,
                               name: str
                               ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_i2c(name, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/i2c", tags=[Tags.data])
async def data_device_i2c(shared: Annotated[dict, Depends(get_shared)],
                          device_id:  int
                          ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_list(device.i2c_names(), info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/alias/{name}", tags=[Tags.data])
async def data_device_alias_name(shared: Annotated[dict, Depends(get_shared)],
                                 device_id:  int,
                                 name: str
                                 ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_alias(name, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/alias", tags=[Tags.data])
async def data_device_alias(shared: Annotated[dict, Depends(get_shared)],
                            device_id:  int
                            ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_list(device.alias_names(), info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/group/{name}", tags=[Tags.data])
async def data_device_group_name(shared: Annotated[dict, Depends(get_shared)],
                                 device_id:  int,
                                 name: str
                                 ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_group(name, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/group", tags=[Tags.data])
async def data_device_group(shared: Annotated[dict, Depends(get_shared)],
                            device_id:  int
                            ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_list(device.group_names(), info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}/{name}", tags=[Tags.data])
async def data_device_name(shared: Annotated[dict, Depends(get_shared)],
                           device_id:  int,
                           name: str
                           ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read(name, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data/device/{device_id}", tags=[Tags.data])
async def data_device(shared: Annotated[dict, Depends(get_shared)],
                      device_id:  int
                      ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    names = []
    if device:
        names.extend(device.bitfield_names())
        names.extend(device.i2c_names())
        names.extend(device.alias_names())
        names.extend(device.group_names())
    bitfields = device.read_list(names, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.post("/data/device/{device_id}", tags=[Tags.data])
async def post_data_device(shared: Annotated[dict, Depends(get_shared)],
                           device_id:  int,
                           names: list[str]
                           ) -> dict[Datatables, list[BitFieldRow]]:
    device = shared.driver.device(device_id)
    bitfields = device.read_list(names, info=True) if device else {}
    return {"data": record_to_row(bitfields)}


@data_router.get("/data", tags=[Tags.data])
async def data_device_list(shared: Annotated[dict, Depends(get_shared)]
                           ) -> dict[Datatables, list[DeviceRecord]]:
    devices = shared.driver.device_list()
    data = []
    for value in devices:
        data.append(value)
    return {"data": data}
