import json
import math
import time

from contextlib import contextmanager
from pathlib import Path
from typing import NewType, Any

from felix_io_api.api import BitFieldInfoRecord, BitFieldRecord, DeviceRecord, get_shared
from .config import get_settings
from .regbitmap import RegBitMap
from .i2c_bitmap import I2CBitMap
from .decoder import decode
from .flxcard_simulator import LOCK
from .util import Util


def convert(value: int, power: float, multiply: float, divide: float, offset: float) -> float | None:
    if value is None or (value == 0 and power < 0):
        return None
    return (float)((pow(value, power) * multiply) / divide) + offset


def twos_complement(value: int, bits: int):
    """compute the 2's complement of int value val"""
    if (value & (1 << (bits - 1))) != 0:  # if sign bit is set e.g., 8bit: 128-255
        value = value - (1 << bits)       # compute negative value
    return value                          # return positive value as is


def linear11(value: int) -> float:
    # Convert Linear11 (Linear-5s-11s) coded value to double
    mantissa = twos_complement((value >> 11) & 0x1f, 5)
    base = twos_complement(value & 0x07FF, 11)
    return (float)(base * pow(2.0, mantissa))


def linear16(value: int) -> float:
    return value * pow(2.0, -12.0)


def to_string(value: int | float | str | None, unit: str | None = None) -> str:
    if value is None:
        return 'n/a'

    if unit:
        return f"{value} {unit}"

    if type(value) is float:
        return f"{value}"

    if type(value) is int:
        return hex(value)

    return value


class FlxCard:

    open_dev = None
    open_device_id = None

    def __init__(self, simulate: bool = False):
        global flxcard
        if simulate:
            from .flxcard_simulator import flxcard
        else:
            from libflxcard_py import flxcard  # noqa: E402

        self.logger = Util.get_logger(__name__, get_settings().log_level)
        n_devices = self.number_of_devices()
        n_cards = self.number_of_cards()

        self.dev = [None] * n_devices

        self.open_level = [0] * n_devices
        self.n_opens = [0] * n_devices

        self.lock_level = [0] * n_devices
        self.n_locks = [0] * n_devices
        self.is_locked = [False] * n_devices

        self.logger.info(f"FlxCard: {n_cards} Cards, {n_devices} Devices")

        # Store the card_ids for all devices
        self.card_id_for_device = [None] * n_devices
        for card_id in range(0, n_cards):
            device_id = self.card_to_device_id(card_id)
            self.card_id_for_device[device_id] = card_id
            if device_id + 1 < n_devices:
                self.card_id_for_device[device_id + 1] = card_id

        for device_id in range(0, n_devices):
            self.logger.info(f"Device #{device_id} is handled by card #{self.card_id_for_device[device_id]}")
            self.dev[device_id] = flxcard()

    def open_card(dev, device_id: int, lock_mask: int):
        if FlxCard.open_dev and FlxCard.open_dev != device_id:
            FlxCard.close_card(FlxCard.open_dev)

        dev.card_open(device_id, lock_mask)
        FlxCard.open_dev = dev
        FlxCard.open_device_id = device_id

    def close_card(dev):
        dev.card_close()
        FlxCard.open_dev = None
        FlxCard.open_device_id = None

    def open(self, device_id: int, name: str, lock: bool = False, now: bool = False) -> tuple[int]:  # noqa: A003
        # self.logger.info(f"{('lock' if lock else 'open')} {device_id}, {name}, {FlxCard.open_device_id}, {self.open_level[device_id]}, {self.lock_level[device_id]}")

        # check for negative open_level and reset
        if self.open_level[device_id] < 0:
            self.logger.warning(f"FlxCard.open({device_id}), level {self.open_level[device_id]} < 0; reset to 0")
            self.open_level[device_id] = 0

        if lock:
            # check for negative lock_level and reset
            if self.lock_level[device_id] < 0:
                self.logger.warning(f"FlxCard.lock({device_id}), level {self.lock_level[device_id]} < 0; reset to 0")
                self.lock_level[device_id] = 0

            # delayed locking for caching
            if now and not self.is_locked[device_id]:
                if self.open_level[device_id] > 0:
                    self.logger.info(f"Closing   device #{device_id} open:{self.open_level[device_id]} lock:{self.lock_level[device_id]}")
                    FlxCard.close_card(self.dev[device_id])
                    # self.open_level[device_id] -= 1

                self.logger.info(f"Locking {LOCK.I2C.value} device #{device_id} open:{self.open_level[device_id]} lock:{self.lock_level[device_id]}")
                FlxCard.open_card(self.dev[device_id], device_id, LOCK.I2C.value)
                self.is_locked[device_id] = True
                self.n_opens[device_id] += 1
                self.n_locks[device_id] += 1

            # increase lock_level
            self.lock_level[device_id] += 1
        else:
            # intermediate open (between locks) should also increase lock_level
            if self.lock_level[device_id] > 0:
                self.lock_level[device_id] += 1

        # open the card if it is not already locked
        if self.open_level[device_id] == 0 and not self.is_locked[device_id]:
            self.logger.info(f"Opening device #{device_id} without lock")
            FlxCard.open_card(self.dev[device_id], device_id, LOCK.NONE.value)
            self.n_opens[device_id] += 1

        # increase the open_level
        self.open_level[device_id] += 1

        assert self.open_level[device_id] >= self.lock_level[device_id]
        # self.logger.info(f"     -> {self.open_level[device_id]}, {self.lock_level[device_id]}")
        return (self.open_level[device_id], self.lock_level[device_id])

    def lock(self, device_id: int, name: str, now: bool = False) -> tuple[int]:
        return self.open(device_id, name, lock=True, now=now)

    def close(self, device_id: int, name: str, unlock: bool = False) -> tuple[int]:
        # self.logger.info(f"{('unlock' if unlock else 'close')}, {device_id}, {name}, {self.open_level[device_id]}, {self.lock_level[device_id]}")

        # Update levels
        self.open_level[device_id] -= 1
        if self.lock_level[device_id] > 0:
            self.lock_level[device_id] -= 1

        # Check on too many close calls
        if self.open_level[device_id] < 0:
            self.logger.info(f"FlxCard.close({device_id}), level {self.open_level[device_id]} < 0; reset to 0")
            self.open_level[device_id] = 0

        # Close the card if close or unlock was called at levels 0
        was_locked = False
        if self.open_level[device_id] == 0 or (self.lock_level[device_id] == 0 and self.is_locked[device_id]):
            was_locked = self.is_locked[device_id]
            self.logger.info(f"Closing {'locked' if was_locked else 'unlocked'} device #{device_id} open:{self.open_level[device_id]} lock:{self.lock_level[device_id]}")
            FlxCard.close_card(self.dev[device_id])
            self.is_locked[device_id] = False

        # Re-open the card if it was open before the lock (if it was really locked)
        if self.open_level[device_id] > 0 and self.lock_level[device_id] == 0 and was_locked:
            self.logger.info(f"Re-opening device #{device_id} without lock")
            FlxCard.open_card(self.dev[device_id], device_id, LOCK.NONE.value)
            self.n_opens[device_id] += 1

        assert self.open_level[device_id] >= self.lock_level[device_id]
        # self.logger.info(f"     -> {self.open_level[device_id]}, {self.lock_level[device_id]}")
        return (self.open_level[device_id], self.lock_level[device_id])

    def unlock(self, device_id: int, name: str) -> tuple[int]:
        return self.close(device_id, name, unlock=True)

    @contextmanager
    def open_flxcard(self, device_id: int, name: str):
        self.open(device_id, name)
        try:
            yield flxcard
        finally:
            self.close(device_id, name)

    @contextmanager
    def lock_flxcard(self, device_id: int, name: str, now: bool = False):
        self.lock(device_id, name, now=now)
        try:
            yield flxcard
        finally:
            self.unlock(device_id, name)

    def reset_statistics(self, device_id: int):
        self.n_opens[device_id] = 0
        self.n_locks[device_id] = 0

    def number_of_opens(self, device_id: int) -> int:
        return self.n_opens[device_id]

    def number_of_locks(self, device_id: int) -> int:
        return self.n_locks[device_id]

    def number_of_cards(self) -> int:
        return flxcard.number_of_cards()

    def number_of_devices(self) -> int:
        return flxcard.number_of_devices()

    def card_to_device_id(self, card_id: int) -> int:
        return flxcard.card_to_device_number(card_id)

    # FIXME we may use a contextmanager here
    # see https://book.pythontips.com/en/latest/context_managers.html
    def read_register(self, device_id: int, name: str) -> int:
        with self.open_flxcard(device_id, name) as card:
            return card.cfg_get_reg(self.dev[device_id], name)

    def write_register(self, device_id: int, name: str, value: int):
        with self.open_flxcard(device_id, name) as card:
            return card.cfg_set_reg(self.dev[device_id], name, value)

    def read_bitfield(self, device_id: int, name: str) -> int:
        with self.open_flxcard(device_id, name) as card:
            return card.cfg_get_option(self.dev[device_id], name)

    def write_bitfield(self, device_id: int, name: str, value: int):
        with self.open_flxcard(device_id, name) as card:
            return card.cfg_set_option(self.dev[device_id], name, value)

    def card_id(self, device_id: int) -> int:
        return self.card_id_for_device[device_id]

    def card_type(self, device_id: int) -> int:
        return self.read_bitfield(device_id, 'CARD_TYPE')

    def read_i2c_byte(self, device_id: int, device_name: str, register_address: int) -> int:
        with self.lock_flxcard(device_id, device_name, now=True) as card:
            return card.read_i2c(self.dev[device_id], device_name, register_address)

    def read_i2c_multiple_bytes(self, device_id: int, device_name: str, register_address: int, nbytes: int) -> list[int]:
        with self.lock_flxcard(device_id, device_name, now=True) as card:
            return card.read_i2c(self.dev[device_id], device_name, register_address, nbytes)

    def write_i2c(self, device_id: int, device_name: str, register_address: int, value: int):
        with self.lock_flxcard(device_id, device_name, now=True) as card:
            card.write_i2c(self.dev[device_id], device_name, register_address, value)


class BitField:

    def __init__(self, regbitmap: RegBitMap, card: FlxCard, device_id: int, name: str):
        self.logger = Util.get_logger(__name__, get_settings().log_level)
        self.regbitmap = regbitmap
        self.card = card
        self.device_id = device_id
        self.name = name

    def get_info(self) -> BitFieldInfoRecord:
        return self.regbitmap.get_bitfield(self.name)

    def can_read(self) -> bool:
        return 'REGMAP_REG_READ' in self.get_info()['type'].split('|')

    def can_write(self) -> bool:
        return 'REGMAP_REG_WRITE' in self.get_info()['type'].split('|')

    def get(self) -> int | float | str | None:
        value = self.get_raw()
        bf = self.get_info()
        if any(key in bf for key in ['power', 'multiply', 'divide', 'offset']):
            # float
            return convert(value, bf.get('power', 1), bf.get('multiply', 1.0), bf.get('divide', 1.0), bf.get('offset', 0.0))

        value = decode(self.name, value)

        # str | int | None
        return value

    def get_raw(self) -> int | None:
        self.card.open(self.device_id, 'get_raw')
        try:
            result = self.card.read_bitfield(self.device_id, self.name)
        except RuntimeError:
            result = None
        finally:
            self.card.close(self.device_id, 'get_raw')
        return result

    def get_unit(self) -> str:
        return self.regbitmap.get_bitfield(self.name).get('unit', None)

    def set(self, value: int):  # noqa: A003
        self.card.open(self.device_id, 'set')
        self.card.write_bitfield(self.device_id, self.name, value)
        self.card.close(self.device_id, 'set')

    def clear_interval(self) -> int:
        bf = self.regbitmap.get_bitfield(self.name)
        return bf.get('clear', 0)

    def alias(self) -> str | None:
        bf = self.regbitmap.get_bitfield(self.name)
        return bf.get('alias', None)

    def group(self) -> list[str] | str | None:
        return self.regbitmap.get_bitfield(self.name).get('group', None)

    def to_string(self) -> str | None:
        return to_string(self.get(),  self.get_unit())

    def record(self, info) -> BitFieldRecord:
        # result: BitFieldRecord = self.get_info().copy()
        value = self.get()
        unit = self.get_unit()

        result = {}
        # result['name'] = self.name
        result['value'] = value
        result['decoded_value'] = to_string(value,  unit)

        raw_value = self.get_raw()
        if result['value'] != raw_value:
            result['raw_value'] = raw_value

        if unit:
            result['unit'] = unit

        if info:
            bf = self.get_info()
            result['info'] = bf.copy()

            # extra fields to copy, make sure they are defined in the api
            for key in ['power', 'multiply', 'divide', 'offset', 'unit', 'clear']:
                if key in bf:
                    result['info'][key] = bf[key]

        return result


# FIXME use inheritance for Bitfield and Register
class Register:

    def __init__(self, regbitmap: RegBitMap, card: FlxCard, device_id: int, name: str):
        self.logger = Util.get_logger(__name__, get_settings().log_level)
        self.regbitmap = regbitmap
        self.card = card
        self.device_id = device_id
        self.name = name

    def get(self) -> int:
        self.card.open(self.device_id, 'get')
        try:
            result = self.card.read_register(self.device_id, self.name)
        except RuntimeError:
            # FIXME should be None
            result = 0
        finally:
            self.card.close(self.device_id, 'get')
        return result

    def set(self, value: int):  # noqa: A003
        self.card.open(self.device_id, 'set')
        self.card.write_register(self.device_id, self.name, value)
        self.card.close(self.device_id, 'set')


I2C = NewType('I2C', None)


class Device:

    def __init__(self, directory: Path, card: FlxCard, device_id: int):
        self.logger = Util.get_logger(__name__, get_settings().log_level)
        self.directory = directory
        self.card = card
        self.device_id = device_id

        self.i2c_device = None

        self.card.open(self.device_id, 'init')

        # Do not use bitfield as self.regbitmap is not initialized here
        regmap_version = decode('REG_MAP_VERSION', self.card.read_bitfield(self.device_id, 'REG_MAP_VERSION'))
        self.logger.info(f"   Regmap Version for device #{device_id}: {regmap_version}")

        major_version = (int)(regmap_version.split('.')[0])
        self.regbitmap = RegBitMap(major_version, self.directory, self.device_id)

        self.card.close(self.device_id, 'init')

    def __del__(self):
        self.card.close(self.device_id, '__del__')

    def number_of_opens(self) -> int:
        return self.card.number_of_opens(self.device_id)

    def number_of_locks(self) -> int:
        return self.card.number_of_locks(self.device_id)

    def name(self):
        return f"Device #{self.id()}"

    def id(self) -> int:  # noqa: A003
        return self.device_id

    def card_id(self) -> FlxCard:
        return self.card.card_id(self.device_id)

    def card_type(self) -> BitField:
        return self.bitfield('CARD_TYPE')

    def bitfield_names(self) -> list[str]:
        return self.regbitmap.bitfield_names()

    def register_names(self) -> list[str]:
        return self.regbitmap.register_names()

    def i2c_names(self) -> list[str]:
        return self.i2c().names()

    def alias_names(self) -> list[str]:
        return get_shared().alias[self.device_id].keys()

    def group_names(self) -> list[str]:
        return get_shared().group[self.device_id].keys()

    def bitfield(self, name: str) -> BitField:
        return BitField(self.regbitmap, self.card, self.device_id, name)

    def register(self, name: str) -> Register:
        return Register(self.regbitmap, self.card, self.device_id, name)

    def i2c(self) -> I2C:
        if not self.i2c_device:
            self.i2c_device = I2C(self.directory, self.device_id, self.card)
        return self.i2c_device

    def record(self) -> DeviceRecord:
        return {
            "name": self.name(),
            "device_id": self.id(),
            "card_id": self.card_id(),
            "card_type": self.card_type().to_string()
        }

    def read_bitfield(self, name: str, info: bool) -> dict[str, BitFieldRecord]:
        data = {}
        if name in self.bitfield_names():
            bitfield = self.bitfield(name)
            if bitfield.can_read():
                data[name] = bitfield.record(info)
        return data

    def write_bitfield(self, name: str, value: int, info: bool) -> dict[str, BitFieldRecord]:
        self.card.open(self.device_id, name)

        data = {}
        if name in self.bitfield_names():
            bitfield = self.bitfield(name)
            if bitfield.can_write():
                bitfield.set(value)

            # need to flag that its not writeable or does not exist

            if bitfield.can_read():
                data[name] = bitfield.record(info)

        self.card.close(self.device_id, name)

        return data

    def read_i2c(self, name: str, info: bool) -> dict[str, BitFieldRecord]:
        result = {}

        device_name, *register_name = name.split(':', 2)
        if register_name and register_name[0]:
            self.card.lock(self.device_id, name)
            r = self.i2c().read(device_name, register_name[0], info)
            if r:
                result[name] = r
            self.card.unlock(self.device_id, name)

        return result

    def read_alias(self, name: str, info: bool) -> dict[str, BitFieldRecord]:
        result = {}

        self.card.lock(self.device_id, name)

        alias_list = [k for k in get_shared().alias[self.device_id].keys() if k == name]
        for alias in alias_list:
            alias_value_list = get_shared().alias[self.device_id][alias]
            for alias_value in alias_value_list:
                device_name, *register_name = alias_value.split(':')
                if register_name:
                    # i2c (read as many as found)
                    result[f'{alias}:{device_name}:{register_name[0]}'] = self.i2c().read(device_name, register_name[0], info)
                else:
                    # register (read only when no i2c values are found)
                    if len(result.keys()) == 0:
                        result[f'{alias}:{alias_value}'] = self.bitfield(alias_value).record(info)

        self.card.unlock(self.device_id, name)

        return result

    def read_group(self, name: str, info: bool) -> dict[str, BitFieldRecord]:
        group_list = get_shared().group[self.device_id].get(name, [])
        return self.read_list(group_list, info)

    def read(self, name: str, info: bool) -> dict[str, BitFieldRecord]:
        bitfield = self.read_bitfield(name, info)
        if bitfield:
            return bitfield

        i2c = self.read_i2c(name, info)
        if i2c:
            return i2c

        alias = self.read_alias(name, info)
        if alias:
            return alias

        group = self.read_group(name, info)
        if group:
            return group

        return {}

    def read_list(self, names: list[str], info: bool) -> dict[str, BitFieldRecord]:
        self.card.lock(self.device_id, 'list')
        data = {}
        for name in names:
            data.update(self.read(name, info))

        self.card.unlock(self.device_id, "list")
        return data

    def write(self, name: str, value: int, info: bool) -> dict[str, BitFieldRecord]:
        bitfield = self.write_bitfield(name, value, info)
        if bitfield:
            return bitfield

        return {}


class Driver:

    def __init__(self, directory: Path, simulate: bool = False):
        self.logger = Util.get_logger(__name__, get_settings().log_level)
        self.directory = directory
        self.card = FlxCard(simulate)

        self.devices = {}

    def number_of_devices(self) -> int:
        return self.card.number_of_devices()

    def number_of_cards(self) -> int:
        return self.card.number_of_cards()

    def card_to_device_id(self, card_id: int) -> int:
        return self.card.card_to_device_id(card_id)

    def create_device(self, device_id: int) -> Device:
        self.devices[device_id] = Device(self.directory, self.card, device_id)
        return self.devices[device_id]

    def device(self, device_id: int) -> Device | None:
        return self.devices.get(device_id, None)

    def device_list(self) -> list[DeviceRecord]:
        data = []
        for device_id in range(0, self.number_of_devices()):
            data.append(self.device(device_id).record())
        return data


class I2C:

    DEFAULT_LENGTH = 2
    DEFAULT_MASK = 0xFFFF

    def __init__(self, directory: Path, device_id: int, card: FlxCard):
        self.device_id = device_id
        self.card = card

        self.bitmap = I2CBitMap(directory, self.card.card_type(self.device_id), self.device_id)

    def register(self, device_name, register_name):
        return self.bitmap.register(device_name, register_name)

    def read_byte(self, device_name: str, register_address: int) -> int:
        return self.card.read_i2c_byte(self.device_id, device_name, register_address)

    def read_multiple_bytes(self, device_name: str, register_address: int, nbytes: int, multi_address: bool) -> list[int]:
        if multi_address:
            return [self.card.read_i2c_byte(self.device_id, device_name, register_address + i) for i in range(nbytes)]
        else:
            return self.card.read_i2c_multiple_bytes(self.device_id, device_name, register_address, nbytes)

    def read_int(self, device_name: str, register_address: int, nbytes: int = 2, reverse: bool = False, multi_address: bool = False) -> int:
        results = self.read_multiple_bytes(device_name, register_address, nbytes, multi_address)

        if reverse:
            results.reverse()

        value = 0
        for result in results:
            value <<= 8
            value |= result

        return value

    def write_byte(self, device_name: str, register_address: int, value: int):
        self.card.write_i2c(self.device_id, device_name, register_address, value)

    def write_sequence(self, device_name: str, sequence: list[list[int]]):
        for entry in sequence:
            if (entry[0] == 'sleep'):
                time.sleep(entry[1])
            else:
                self.write_byte(device_name, entry[0], entry[1])

    def write_index(self, device_name: str, entry: list[int], index):
        if entry:
            self.write_byte(device_name, entry[0], index + entry[1])

    def get_mask(self, device_name: str, register_name: str, i2c_type: str):
        register = self.register(device_name, register_name)

        mask = register.get('mask', I2C.DEFAULT_MASK)
        match i2c_type:
            case 'word':
                mask &= 0xFFFF
            case 'rword':
                mask &= 0xFFFF
            case 'int':
                mask &= 0xFFFF
            case 'rint':
                mask &= 0xFFFF
            case 'byte':
                mask &= 0xFF
        return mask

    def read_integer(self, device_name: str, register_name: str) -> BitFieldRecord:
        register = self.register(device_name, register_name)

        sequence = register.get('write_bytes', [])
        self.write_sequence(device_name, sequence)

        result = []
        result_raw_value = []
        base_address = register['address']
        i2c_type = register.get('type', 'byte')
        array = register.get('array', 1)
        mask = self.get_mask(device_name, register_name, i2c_type)
        length = register.get('length', I2C.DEFAULT_LENGTH)

        for index in range(0, array):
            entry = register.get('write_index', [])
            self.write_index(device_name, entry, index)

            value = None
            match i2c_type:
                case 'word':
                    value = self.read_int(device_name, base_address + (index * 2), length, multi_address=True) & mask
                case 'rword':
                    value = self.read_int(device_name, base_address + (index * 2), length, reverse=True, multi_address=True) & mask
                case 'int':
                    value = self.read_int(device_name, base_address + index, length) & mask
                case 'rint':
                    value = self.read_int(device_name, base_address + index, length, reverse=True) & mask
                case 'byte':
                    value = self.read_byte(device_name, base_address + index) & mask

            raw_value = value
            match register.get('convert', 'default'):
                case 'linear11':
                    value = linear11(value)
                case 'linear16':
                    value = linear16(value)
                case 'byte_twos_complement':
                    value = twos_complement(value, 8)
                case _:
                    if any(key in register for key in ['power', 'multiply', 'divide', 'offset']):
                        power = register.get('power', 1)
                        multiply = register.get('multiply', 1.0)
                        divide = register.get('divide', 1.0)
                        offset = register.get('offset', 0)
                        value = convert(value, power, multiply, divide, offset)

            # check to add extra registers if needed 26.5 where 26 is in the main register and 0.5 in another
            add_register = register.get('add_register', None)
            if add_register:
                value += self.read_integer(device_name, add_register)['value']

            result.append(value)
            result_raw_value.append(raw_value)

        end_sequence = register.get('end_write_bytes', [])
        self.write_sequence(device_name, end_sequence)

        if register.get('reverse', False):
            result.reverse()
            result_raw_value.reverse()

        unit = register.get('unit', '' if type(result[0]) is float else None)
        value = result if 'array' in register else result[0]
        raw_value = None
        if result_raw_value != result:
            raw_value = result_raw_value if 'array' in register else result_raw_value[0]

        if type(value) is list:
            decoded_value = []
            for v in value:
                decoded_value.append(to_string(v, unit))
        else:
            decoded_value = to_string(value, unit)

        result = {}
        result["value"] = value
        result["decoded_value"] = decoded_value
        if unit:
            result["unit"] = unit
        if raw_value:
            result["raw_value"] = raw_value
        return result

    def read_string(self, device_name: str, register_name: str) -> str:
        register = self.register(device_name, register_name)

        sequence = register.get('write_bytes', [])
        self.write_sequence(device_name, sequence)

        address = register['address']
        length = register.get('length', 1)
        multi_address = register.get('type', 'str') == 'str'
        result = ''.join([chr(x) for x in self.read_multiple_bytes(device_name, address, length, multi_address)])

        return result.split("\x00")[0].strip()

    def read(self, device_name: str, register_name: str, info: bool, ignore_sleep: bool = False) -> BitFieldRecord | None:
        register = self.register(device_name, register_name)

        if not register:
            return None

        self.card.lock(self.device_id, device_name)

        cache_key = f"i2c:c{self.card.card_id}:{device_name}:{register_name}"
        shared = get_shared()
        cached_value = shared.cache.get(cache_key) if shared.cache else None
        if cached_value:
            # read from cache
            result = json.loads(cached_value)
            result['ttl'] = shared.cache.ttl(cache_key)

        else:
            # actual read

            # sleep if the register specifies to read twice and sleep inbetween.
            # ignore the sleep if a group/device sleep on a device level is done
            sleep = register.get('sleep', None)
            if not ignore_sleep and sleep:
                match register.get('type', 'byte'):
                    case 'int' | 'rint' | 'rword' | 'word' | 'byte':
                        self.read_integer(device_name, register_name)
                    case 'str' | 'rstr':
                        self.read_string(device_name, register_name)
                time.sleep(sleep)

            match register.get('type', 'byte'):
                case 'int' | 'rint' | 'rword' | 'word' | 'byte':
                    result = self.read_integer(device_name, register_name)
                case 'str' | 'rstr':
                    string = self.read_string(device_name, register_name)
                    result = {
                        "value": string,
                        "decoded_value": string
                    }

            if shared.cache:
                # Store the response in Redis with a TTL
                shared.cache.set(cache_key, json.dumps(result), ex=shared.cache_ttl)

        if info:
            result["info"] = dict(register)
            result["info"]["desc"] = "TBS"
            result["info"]["type"] = result["info"].get("type", "byte")
            result["info"]["mask"] = self.get_mask(device_name, register_name, result["info"]["type"])
            result["info"]["bf_hi"] = int(math.log2(result["info"]["mask"] + 1)) - 1
            result["info"]["bf_lo"] = 0
            result["info"]["fw_modes"] = "n/a"
            result["info"]["endpoints"] = "n/a"

        self.card.unlock(self.device_id, device_name)
        return result

    def max_sleep(self, device_name: str) -> float:
        sleep = -1.0
        for register_name in self.bitmap.register_names(device_name):
            sleep = max(self.register(device_name, register_name).get('sleep', -1.0), sleep)
        return sleep

    def detect_device(self, device_name: str) -> dict[str, str | bool | int]:
        self.card.lock(self.device_id, device_name)

        result = {}
        chip = self.bitmap.device(device_name).get('chip', '')
        result['chip'] = chip
        match chip:
            # 712
            case 'MiniPOD AFBR-814':
                result['Name'] = self.read(device_name, 'Name')
                result['Found'] = result['Name'] == 'AVAGO'
            case 'SI5345':
                result['PN_BASE'] = self.read(device_name, 'PN_BASE')
                result['Found'] = result['PN_BASE'] == 0x5345
            case 'LTC2991' | 'ADN2814' | 'TCA6408A' | 'SIS53154':
                # No ID, assume its there
                result['Found'] = True

            # 182
            case 'ADM1066':
                result['manufacturer_id'] = self.read(device_name, 'manufacturer_id')
                result['Found'] = result['manufacturer_id'] == 0x41
            case 'ADM1266':
                result['device_id'] = self.read(device_name, 'device_id')
                result['device_rev'] = self.read(device_name, 'device_rev')
                result['Found'] = result['device_id'] == 3 and result['device_rev'] == 8  # NOTE: lengths as we cannot use Block Read
            case 'LTM4700':
                result['MFR_ID'] = self.read(device_name, 'MFR_ID')
                result['MFR_MODEL'] = self.read(device_name, 'MFR_MODEL')
                result['Found'] = result['MFR_ID'] == 3 and result['MFR_MODEL'] == 7  # NOTE: lengths as we cannot use Block Read
            case 'INA226':
                result['manufacturer_id'] = self.read(device_name, 'manufacturer_id')
                result['die_id'] = self.read(device_name, 'die_id')
                result['Found'] = result['manufacturer_id'] == 0x5449 and result['die_id'] == 0x2260
            case 'TMP435':
                result['manufacturer_id'] = self.read(device_name, 'manufacturer_id')
                result['device_id'] = self.read(device_name, 'device_id')
                result['Found'] = result['manufacturer_id'] == 0x55 and result['device_id'] == 0x35  # doc says 0x31
            case 'FIREFLY_TX':
                result['part no'] = self.read(device_name, 'part no')
                result['cern part no'] = self.read(device_name, 'cern part no')
                result['Found'] = result['cern part no'].startswith('CERN') or result['part no'] == 'B042803004170'  # not sure about part no
            case 'FIREFLY_RX':
                result['part no'] = self.read(device_name, 'part no')
                result['cern part no'] = self.read(device_name, 'cern part no')
                result['Found'] = result['cern part no'].startswith('CRRN') or result['part no'] == 'B042803004170'  # not sure about part no
            case 'FIREFLY_TX_RX':
                result['part no'] = self.read(device_name, 'part no')
                result['cern part no'] = self.read(device_name, 'cern part no')
                result['Found'] = result['part no'] == 'B042803004170'
            case 'SI570' | 'TCA6424':
                # No ID, assume its there
                result['Found'] = True

            case '':
                result['Found'] = False
            case _:
                result['Found'] = False

        self.card.unlock(self.device_id, device_name)
        return result

    def names(self) -> list[str]:
        names = []
        for device_name in self.bitmap.device_names():
            for register_name in self.bitmap.register_names(device_name):
                names.append(f'{device_name}:{register_name}')
        return names

    # def group_names(self) -> list[str]:
    #     return self.bitmap.group_names()

    # def device_names_in_group(self, group: str) -> list[str]:
    #     return self.bitmap.device_names_in_group(group)

    def device_names(self) -> list[str]:
        return self.bitmap.device_names()

    def register_names(self, device_name: str) -> list[str]:
        return self.bitmap.register_names(device_name)

    def detect(self) -> dict[str, dict[str, Any]]:
        self.card.lock(self.device_id, "detect")

        result = {}

        for device_name in self.bitmap.device_names():
            result[device_name] = self.detect_device(device_name)

        self.card.unlock(self.device_id, "detect")

        return result
