import json
import yaml

from enum import Enum

from .util import Util
from .config import get_settings


class LOCK(Enum):
    NONE = 0
    DMA0 = 1
    DMA1 = 2
    I2C = 4
    FLASH = 8
    ELINK = 16
    ALL = 0xffffffff


#
# NOTE: only one device can be open/locked at any time
#
class flxcard:

    N_CARDS = 3
    N_DEVICES = 5
    open_device = None

    def __init__(self):
        self.logger = Util.get_logger(__name__, get_settings().log_level)

        self.device_id = None
        self.lock_mask = None

        self.values = None
        self.i2c = None

    def number_of_cards() -> int:
        return flxcard.N_CARDS

    def number_of_devices() -> int:
        return flxcard.N_DEVICES

    def card_to_device_number(card_id: int) -> int:
        match card_id:
            case 0:  # 709 (d0)
                return 0
            case 1:  # 712 (d1 and d2)
                return 1
            case 2:  # 182 (d3 and d4)
                return 3
            case _:  # invalid
                return -1

    def card_open(self, n: int, lock_mask: LOCK, read_config: bool = False, ignore_version: bool = False):
        self.logger.info(f"flxcard.open(device_id: {n}, lock_mask: {lock_mask}, read_config: {read_config}, ignore_version: {ignore_version})")

        assert flxcard.open_device is None, f"Trying to open device {n}, while device {flxcard.open_device} is still open"
        assert n is not None, "Trying to open device None"
        assert lock_mask is not None, "Trying to open device with lock_mask None"
        flxcard.open_device = n

        self.device_id = n
        self.lock_mask = lock_mask

        if self.values is None:
            project_root = Util.project_root()
            directory = project_root / "share" / "felix-io" / "data"
            json_path = directory / f"device{self.device_id}.json"
            with json_path.open() as stream:
                self.logger.info(f"Reading simulated Register values from: {json_path}")
                self.values = json.load(stream)

            card_type = self.values["CARD_TYPE"]
            self.logger.info(f"Card type: {card_type}")

            self.i2c = None
            yaml_path = directory / f"{card_type}.yaml"
            if yaml_path.is_file():
                with yaml_path.open() as stream:
                    self.logger.info(f"Reading simulated I2C values from: {yaml_path}")
                    self.i2c = yaml.safe_load(stream)
            else:
                self.logger.info(f"NO I2C values found at: {yaml_path}")

    def card_close(self):
        self.logger.info(f"flxcard.close() device_id {self.device_id}")
        assert self.device_id is not None, "Device ID is None"
        assert self.device_id == flxcard.open_device, f"Trying to close a different id {self.device_id} than the open one {flxcard.open_device}"
        flxcard.open_device = None

    def get_lock_mask(self, n: int) -> LOCK:
        return self.lock_mask

    # NOTE: get and set register do not mix in the simulator with get and set bitfields
    #       for the actual flxcard they work fine together
    def cfg_get_reg(self, key: str) -> int:
        assert self.device_id == flxcard.open_device, f"Accessing device {self.device_id} while device {flxcard.open_device} is open"

        if key in self.values.keys():
            return self.values[key]
        else:
            raise RuntimeError(f"Register/Bitfield {key} not available for device {self.device_id}")

    def cfg_set_reg(self, key: str, value: int):
        assert self.device_id == flxcard.open_device, f"Accessing device {self.device_id} while device {flxcard.open_device} is open"

        if key in self.values.keys():
            self.values[key] = value
        else:
            raise RuntimeError(f"Register/Bitfield {key} not available for device {self.device_id}")

    def cfg_get_option(self, key: str) -> int:
        assert self.device_id == flxcard.open_device, f"Accessing device {self.device_id} while device {flxcard.open_device} is open"

        if key in self.values.keys():
            return self.values[key]
        else:
            raise RuntimeError(f"Register/Bitfield {key} not available for device {self.device_id}")

    def cfg_set_option(self, key: str, value: int):
        assert self.device_id == flxcard.open_device, f"Accessing device {self.device_id} while device {flxcard.open_device} is open"

        if key in self.values.keys():
            self.values[key] = value
        else:
            raise RuntimeError(f"Register/Bitfield {key} not available for device {self.device_id}")

    def read_i2c(self, device_name: str, reg_address: int, nbytes: int = 0) -> int | list[int]:
        assert self.device_id == flxcard.open_device, "Accessing device {self.device_id} while device {flxcard.open_device} is open"
        assert self.lock_mask == LOCK.I2C.value, f"Device locked with {self.lock_mask} while {LOCK.I2C.value} expected"

        if self.i2c is None:
            return 0

        return self.i2c.get(device_name, {}).get(reg_address, {}).get('data', 0)

    def write_i2c(self, device_name: str, reg_address: int, value: int = None, data: list[int] = None) -> None:
        assert self.device_id == flxcard.open_device, "Accessing device {self.device_id} while device {flxcard.open_device} is open"
        assert self.lock_mask == LOCK.I2C.value, f"Device locked with {self.lock_mask} while {LOCK.I2C.value} expected"

        # ignore
        pass
